package org.gcube.data.spd.wormsplugin.capabilities;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.spd.plugin.fwk.Properties;
import org.gcube.data.spd.plugin.fwk.Property;
import org.gcube.data.spd.plugin.fwk.capabilities.ClassificationCapability;
import org.gcube.data.spd.plugin.fwk.exceptions.ExternalRepositoryException;
import org.gcube.data.spd.plugin.fwk.exceptions.IdNotValidException;
import org.gcube.data.spd.plugin.fwk.exceptions.MethodNotSupportedException;
import org.gcube.data.spd.plugin.fwk.model.CommonName;
import org.gcube.data.spd.plugin.fwk.model.TaxonomyItem;
import org.gcube.data.spd.plugin.fwk.model.TaxonomyStatus;
import org.gcube.data.spd.plugin.fwk.model.TaxonomyStatus.Status;
import org.gcube.data.spd.plugin.fwk.util.ElementProperty;
import org.gcube.data.spd.plugin.fwk.writers.ObjectWriter;
import org.gcube.data.spd.wormsplugin.Utils;
import org.gcube.data.spd.wormsplugin.WormsPlugin;

import aphia.v1_0.AphiaRecord;
import aphia.v1_0.Source;
import aphia.v1_0.Vernacular;

public class ClassificationCapabilityImpl extends ClassificationCapability {

	GCUBELog logger = new GCUBELog(ClassificationCapabilityImpl.class);

	@Override
	public Set<Properties> getSupportedProperties() {
		return Collections.emptySet();
	}

	@Override
	public List<TaxonomyItem> retrieveTaxonChildsByTaxonId(String id) {

		//		logger.trace(id);
		List<TaxonomyItem> list = new ArrayList<TaxonomyItem>(); 	
		try {
			AphiaRecord[] records;
			final int offsetlimit=50;
			int offset =1;
			do{
				records =  WormsPlugin.binding.getAphiaChildrenByID(Integer.parseInt(id), offset);

				if (records!=null){
					//										logger.debug(records.length);
					for (AphiaRecord record : records){
						TaxonomyItem item = createItem(record, false);
						if (item!=null)
							list.add(item);
					}
				}
				offset+=offsetlimit;
			} while (records!=null && records.length==offsetlimit);

		} catch (Throwable e) {
			logger.error("error contacting worms service", e);
		}

		return list;
	}




	public void retrieveTaxa(ObjectWriter<TaxonomyItem> writer, String word, String type) {

		Set<Integer> hash = new HashSet<Integer>();

		try {
			AphiaRecord[] records = null;

			final int offsetlimit=50;
			int offset =1;
			do{
				if (type.equals("vernacular"))
					records = WormsPlugin.binding.getAphiaRecordsByVernacular(word, true, offset);
				else if (type.equals("scientific"))
					records = WormsPlugin.binding.getAphiaRecords(word, true, false, false, offset);

				if (records!=null){
					for (AphiaRecord record : records){

						if (hash.contains(record.getAphiaID())){
							continue;
						}

						hash.add(record.getAphiaID());

						TaxonomyItem item = createItem(record, true);
						if ((item != null) && (writer.isAlive()))
							writer.write(item);

					}
				}


				offset+=offsetlimit;
			} while (records!=null && records.length==offsetlimit);

		} catch (Throwable e) {
			logger.error("error contacting worms service", e);
		}
		finally{
			writer.close();
		}
	}

	//create TaxonomyItem
	private TaxonomyItem createItem(AphiaRecord record, Boolean flag) throws RemoteException {
		TaxonomyItem item = null;
		String scientificname = record.getScientificname();
		if (scientificname != null ){

			item = new TaxonomyItem(record.getAphiaID()+"");

			item.setScientificName(record.getScientificname());
			item.setAuthor(record.getAuthority());
			item.setLsid(record.getLsid());

			item.setCredits(Utils.createCredits());			
			item.setCitation(record.getCitation());

			item.setRank(record.getRank());		
			List<CommonName> listCommNames = new ArrayList<CommonName> ();

			Vernacular[] vernaculars = WormsPlugin.binding.getAphiaVernacularsByID(record.getAphiaID());
			if (vernaculars!=null){
				for (Vernacular vernacular : vernaculars) {			
					if (vernacular.getLanguage_code()!=null){
						CommonName a = new CommonName(vernacular.getLanguage(),vernacular.getVernacular());
						listCommNames.add(a);		
					}
				}
			}
			item.setCommonNames(listCommNames);		

			try{
				if (record.getStatus().equals("accepted"))
					item.setStatus(new TaxonomyStatus("accepted", Status.ACCEPTED));
				else if (record.getStatus().equals("unaccepted") && WormsPlugin.binding.getAphiaRecordByID(record.getValid_AphiaID()).getStatus().equals("accepted")){
					//							logger.trace(WormsPlugin.binding.getAphiaSynonymsByID(record.getValid_AphiaID()));						
					item.setStatus(new TaxonomyStatus(Status.SYNONYM, record.getValid_AphiaID()+"", "synonym"));}
				else
					item.setStatus(new TaxonomyStatus(record.getStatus(), Status.UNKNOWN));
			}catch (Exception e) {
				item.setStatus(new TaxonomyStatus(record.getStatus(), Status.UNKNOWN));
			}

			if (flag){
				try{
					item.setParent(Utils.retrieveTaxonomy( WormsPlugin.binding.getAphiaClassificationByID(record.getAphiaID()), record.getAphiaID()));
				}catch (Exception e) {
					item.setParent(null);
				}
			}
			else 
				item.setParent(null);

			Source[] sources = null;
			if ((sources= WormsPlugin.binding.getSourcesByAphiaID(record.getAphiaID()))!=null){
				for (Source source : sources){
					//			logger.trace(source.getReference());
					if (source==null)
						continue;
					if (source.getReference()!=null){

						StringBuilder p = new StringBuilder();
						p.append(source.getReference());
						if (source.getLink()!=null){
							p.append(", available online at ");
							p.append(source.getLink());
						}
						if (source.getUrl()!=null){
							p.append(", details: ");
							p.append(source.getUrl());
						}				
						ElementProperty property = new ElementProperty(source.getUse(), p.toString());
						item.addProperty(property);				
					}
				}
			}
		}

		return item;
	}

	@Override
	public void searchByCommonName(String commonName,
			ObjectWriter<TaxonomyItem> writer, Property... properties) {
		retrieveTaxa(writer, commonName, "vernacular");

	}

	@Override
	public void searchByScientificName(String scientificName,
			ObjectWriter<TaxonomyItem> writer, Property... properties) {
		retrieveTaxa(writer, scientificName, "scientific");

	}

	@Override
	public void retrieveTaxonByIds(Iterator<String> ids, ObjectWriter<TaxonomyItem> writer) {

		try{
			while(ids.hasNext()) {

				String id = ids.next(); 
				//				logger.trace("Retrive taxon by id " + id);
				TaxonomyItem tax = retrieveTaxonById(id);
				if ((tax != null) && (writer.isAlive()))
					writer.write(tax);
				else
					break;
			}

		} catch (IdNotValidException e) {
			logger.error("IdNotValidn", e);
		} catch (ExternalRepositoryException e) {
			logger.error("ExternalRepositoryException", e);
		} finally {
			writer.close();	

		}

	}


	@Override
	public TaxonomyItem retrieveTaxonById(String id)
			throws IdNotValidException, ExternalRepositoryException {

		TaxonomyItem tax = null;
		//		logger.trace("Retrive taxon by id " + id);
		AphiaRecord record = null;

		try {
			record = WormsPlugin.binding.getAphiaRecordByID(Integer.parseInt(id));
			if (record!=null){				
				tax = createItem(record, true);
			}
		} catch (NumberFormatException e) {
			logger.error("NumberFormatException", e);
		} catch (RemoteException e) {
			logger.error("Remote Error", e);
		}

		return tax;

	}


	@Override
	public void getSynonymnsById(ObjectWriter<TaxonomyItem> writer, String id)
			throws IdNotValidException, MethodNotSupportedException,
			ExternalRepositoryException {
		try{			

			AphiaRecord[] records = null;
			if ((records = WormsPlugin.binding.getAphiaSynonymsByID(Integer.parseInt(id)))!=null){
				for (AphiaRecord record : records){

					TaxonomyItem tax = createItem(record, true);
					if ((tax != null) && (writer.isAlive()))
						writer.write(tax);
					else
						break;
				}	
			}
		}catch (Exception e) {
			logger.error("General Error", e);
		}
	}
}
