package org.gcube.data.spd.flora;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.gcube.common.core.resources.GCUBERuntimeResource;
import org.gcube.common.core.resources.runtime.AccessPoint;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.spd.flora.dbconnection.ConnectionPool;
import org.gcube.data.spd.flora.dbconnection.ConnectionPoolException;
import org.gcube.data.spd.flora.dbconnection.RSItem;
import org.gcube.data.spd.plugin.fwk.AbstractPlugin;
import org.gcube.data.spd.plugin.fwk.Capabilities;
import org.gcube.data.spd.plugin.fwk.Property;
import org.gcube.data.spd.plugin.fwk.SearchTypes;
import org.gcube.data.spd.plugin.fwk.capabilities.ClassificationCapability;
import org.gcube.data.spd.plugin.fwk.model.DataProvider;
import org.gcube.data.spd.plugin.fwk.model.DataSet;
import org.gcube.data.spd.plugin.fwk.model.ResultItem;
import org.gcube.data.spd.plugin.fwk.model.Taxon;
import org.gcube.data.spd.plugin.fwk.util.RepositoryInfo;
import org.gcube.data.spd.plugin.fwk.writers.ObjectWriter;


public class FloraPlugin extends AbstractPlugin{


	public static String jdbc;
	public static String username ;
	public static String password;
	public static String baseurl;


	public static final String credits = "This information object has been generated via the Species Product Discovery service on XDATEX by interfacing with Lista de Espécies da Flora do Brasil (http://floradobrasil.jbrj.gov.br/)";
	public static final String citation = "Accessed through: Lista de Espécies da Flora do Brasil - Jardim Botânico do Rio de Janeiro at http://floradobrasil.jbrj.gov.br/2012/ on ";
	public static final String dumpDb = "/createFloraDb.sql";
	public static final String tableName = "flora";
	public static final String driver = "org.postgresql.Driver";

	GCUBELog logger = new GCUBELog(FloraPlugin.class);

	@Override
	public void initialize(GCUBERuntimeResource res) throws Exception {
		for (AccessPoint ap:res.getAccessPoints()) {
			if (ap.getEntryname().equals("rest")) {
				baseurl = ap.getEndpoint();
			}
			else if (ap.getEntryname().equals("jdbc")) {
				jdbc = ap.getEndpoint();	
				username = ap.getUsername();
				password = ap.getPassword();			
			}
		}
		if ((!Utils.SQLTableExists(FloraPlugin.tableName) || (!Utils.SQLTableExists("updates")))){
			System.out.println("Create tables");
			if (Utils.createDB())
				logger.trace("Completed.");			
		}

		long update = Utils.lastupdate();
		new UpdateThread(update);
	}


	@Override
	public void update(GCUBERuntimeResource res) throws Exception {
		for (AccessPoint ap:res.getAccessPoints()) {
			if (ap.getEntryname().equals("rest")) {
				baseurl = ap.getEndpoint();
			}
			else if (ap.getEntryname().equals("jdbc")) {
				jdbc = ap.getEndpoint();	
				username = ap.getUsername();
				password = ap.getPassword();			
			}
		}
		super.update(res);
	}


	@Override
	public String getDescription() {
		return ("Brazilian Flora Plugin");
	}

	@Override
	public String getRepositoryName() {
		return ("BrazilianFlora");
	}

	@Override
	public void shutdown() throws Exception {
		//		Discover.shutdown();
	}


	@SuppressWarnings("serial")
	@Override
	public Set<Capabilities> getSupportedCapabilities() {
		return new HashSet<Capabilities>(){{add(Capabilities.Classification);}};
	}

	@Override
	public Set<SearchTypes> getSupportedSearch() {
		return Collections.singleton(SearchTypes.ScientificName);
	}

	@Override
	public ClassificationCapability getClassificationInterface() {
		return new ClassificationCapabilityImpl();
	}


	@Override
	public void searchByCommonName(String word,
			ObjectWriter<ResultItem> writer, Property... properties) {
	}


	@Override
	public void searchByScientificName(String scientificName,
			ObjectWriter<ResultItem> writer, Property... properties){

		ResultSet results = null;
		try{
			results = createRSNames(scientificName);
			if (results!=null){

				while(results.next()) {	

					RSItem item = new RSItem();
					item.setId(results.getString(1));
					item.setScientific_name(results.getString(2));
					item.setRank(results.getString(3));
					item.setParentId(results.getString(4));
					item.setAuthor(results.getString(5));

					ResultItem rs = null;

					try{
						rs = createResultItem(item);
					}catch (Exception e) {
						logger.error("Error retrieving information ", e);
					}

					if (writer.isAlive()){
						if (rs!=null)
							writer.write(rs);
					}
					else
						break;
				}
			}
		} catch (SQLException e) {
			logger.error("sql Error", e);
		}finally
		{
			writer.close();
			try {
				if (results != null)
					results.close();
			} catch (SQLException ex) {
				logger.error("sql Error", ex);
			}
		}
	}



	/**
	 * get ResultSet by scientificName
	 */
	private ResultSet createRSNames(String scientificName) {
		ResultSet results = null;	

		ConnectionPool pool = null;

		try {
			pool = ConnectionPool.getConnectionPool();
			String term = "%" + scientificName + "%";

			String query = "select id, scientific_name, rank, id_parent, citation from "+ tableName + " where UPPER(scientific_name) like UPPER(?)";
			results =  pool.selectPrestatement(query, term);
		} catch (ConnectionPoolException e) {
			logger.error("ConnectionPoolException",e);
		}

		return results;

	}


	private ResultItem createResultItem(RSItem rs) {

		String id = rs.getId();
		String ScName = rs.getScientific_name();
		String rank = rs.getRank();
		String idParent = rs.getParentId();
		String author = rs.getAuthor();

		ResultItem item = new ResultItem(id, ScName);

		item.setCitation(Utils.createCitation());
		item.setCredits(Utils.createCredits());

		item.setRank(rank);
		item.setAuthor(author);
		//	    		rs.setProvider("Brazilian Flora");

		DataSet dataSet = new DataSet("floraid");
		dataSet.setName("Brazilian Flora");
		DataProvider dp = new DataProvider("floraid");
		dp.setName("Brazilian Flora");
		dataSet.setDataProvider(dp);
		item.setDataSet(dataSet);

		if ((rank != null) && (idParent!= null)) {
			RSItem parentRS = findTaxonById(idParent);
			if (parentRS!=null){
				Taxon parent = createTaxonByRSItem(parentRS);
				item.setParent(parent);
			}
		}

		return item;	  
	}


	public Taxon createTaxonByRSItem(RSItem rs){
		Taxon parent = null;
		RSItem parentRs = null;
		Taxon t = new Taxon(rs.getId());
		t.setRank(rs.getRank());
		t.setScientificName(rs.getScientific_name());
		t.setAuthor(rs.getAuthor());     
		if  (rs.getParentId() != null)    {    	
			parentRs = findTaxonById(rs.getParentId());
			parent = createTaxonByRSItem(parentRs);
		}

		t.setParent(parent);
		return t;
	}


	/**
	 * Find taxonomy 
	 */
	public RSItem findTaxonById(String id_record) {		
		//		Taxon t = null;

		ResultSet results = null;
		RSItem item = null;
		ConnectionPool pool = null;
		try {
			pool = ConnectionPool.getConnectionPool();	

			String query = "select id_parent, rank, scientific_name, citation from "+ FloraPlugin.tableName + " where id = ?";
			results =  pool.selectPrestatement(query, id_record);

			if(results.next()) {	  
				item = new RSItem();
				item.setId(id_record);
				item.setParentId(results.getString(1));
				item.setRank(results.getString(2));
				item.setScientific_name(results.getString(3));
				item.setAuthor(results.getString(4));
			}
		}
		catch (SQLException sqlExcept) {
			logger.error("sql Error",sqlExcept);
		} catch (ConnectionPoolException e) {
			logger.error("ConnectionPoolException",e);
		} finally{
			try {
				results.close();
			} catch (SQLException e) {
				logger.error("sql Error",e);
			}
		}
		//        logger.trace("returning taxon");
		return item;		
	}

	@Override
	public RepositoryInfo getRepositoryInfo() {
		RepositoryInfo info = new RepositoryInfo(
				"http://floradobrasil.jbrj.gov.br/2012/imgs/logo_fdb_2012.gif", 
				"http://floradobrasil.jbrj.gov.br/",
				"The List of Species of Brazilian Flora is a project coordinated by the Botanical Garden of Rio de Janeiro in partnership with the CRIA. CRIA is responsible for the development and maintenance of the information system. The whole process of validation and inclusion of new data is done online by a network of over 400 taxonomists from Brazil and abroad. The online public version was completed in May 2010."				);
		return info;
	}

}



