package org.gcube.data.publishing.gis.publisher;

import static org.gcube.data.streams.dsl.Streams.convert;
import static org.gcube.data.streams.dsl.Streams.pipe;
import gr.uoa.di.madgik.grs.record.GenericRecord;

import java.net.URI;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.UUID;

import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.porttypes.GCUBEPortType;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.types.MapItemType;
import org.gcube.common.core.types.StringMap;
import org.gcube.common.core.types.VOID;
import org.gcube.data.gis.publisher.stubs.DestinationInternalFault;
import org.gcube.data.gis.publisher.stubs.DestinationTypeNotSupportedFault;
import org.gcube.data.gis.publisher.stubs.DestinationUnreachableFault;
import org.gcube.data.gis.publisher.stubs.GISType;
import org.gcube.data.gis.publisher.stubs.IncorrectDataTypeFault;
import org.gcube.data.gis.publisher.stubs.PublishDataRequestType;
import org.gcube.data.gis.publisher.stubs.PublisherPortType;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.CSquarePoint;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.CoordinatePoint;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.GISData;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.GISFile;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.StreamedGISData;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.faults.DataParsingException;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.faults.NotSupportedDataTypeException;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.faults.RepositoryTypeNotSupportedException;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.faults.UnreachableDestinationException;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.requests.GISRequestConfiguration;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.types.GISDataType;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.types.GISFileType;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.types.GISRepositoryType;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.utils.GISXstream;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.utils.ResultGenerator;
import org.gcube.data.publishing.gis.publisher.plugins.InteractionWrapper;
import org.gcube.data.publishing.gis.publisher.reports.StoredReport;
import org.gcube.data.publishing.gis.publisher.stubs.utils.RSWrapper;
import org.gcube.data.streams.Stream;

public class PublisherPT extends GCUBEPortType implements PublisherPortType{

	/**{@inheritDoc}*/
	@Override	protected GCUBEServiceContext getServiceContext() {
		return ServiceContext.getContext();
	}


	@Override
	public String publishData(PublishDataRequestType request)
	throws RemoteException, GCUBEFault, DestinationUnreachableFault,
	IncorrectDataTypeFault, DestinationInternalFault,
	DestinationTypeNotSupportedFault {
		try {
			GISRequestConfiguration config=(GISRequestConfiguration) GISXstream.get().fromXML(request.getSerializedConfiguration());
			logger.debug("Serving request "+config+", data URI : "+request.getDataLocator()+", data type "+request.getDataType());			
			GISData data=null;
			if(request.getDataType().equals(GISType.COORD_STREAM)){
				Stream<CoordinatePoint> stream=pipe(convert(new URI(request.getDataLocator())).of(GenericRecord.class).withDefaults()).through(new ResultGenerator<CoordinatePoint>());
				data=new StreamedGISData<CoordinatePoint>(stream, CoordinatePoint.class);
			}else if(request.getDataType().equals(GISType.CSQUARE_STREAM)){				
				Stream<CSquarePoint> stream=pipe(convert(new URI(request.getDataLocator())).of(GenericRecord.class).withDefaults()).through(new ResultGenerator<CSquarePoint>());
				data=new StreamedGISData<CSquarePoint>(stream, CSquarePoint.class);
			}else {				
				GISFileType fileType=GISFileType.NETCDF;
				if(request.getDataType().equals(GISType.GEOTIFF))fileType=GISFileType.GEOTIFF;
				data=new GISFile(RSWrapper.getStreamFromLocator(new URI(request.getDataLocator())), fileType);
			}
			GCUBEScope scope=ServiceContext.getContext().getScope();
			InteractionWrapper wrapper=PluginManager.get(scope).getInteractionPlugin(config.getRepositoryDestination());

			return wrapper.publishData(data, config).getId().toString();
		} catch (RepositoryTypeNotSupportedException e) {
			logger.error("Repository type not found ",e);
			throw new DestinationTypeNotSupportedFault();
		} catch (DataParsingException e) {
			logger.error("Unable to parse data",e);
			throw new IncorrectDataTypeFault();
		} catch (NotSupportedDataTypeException e) {
			logger.error("Plugin doesn't support passed data type");
			throw new IncorrectDataTypeFault();
		} catch (UnreachableDestinationException e) {
			logger.error("Selected destination is down",e);
			throw new DestinationUnreachableFault();
		} catch (Throwable t) {
			logger.error("Unexpected Exception", t);
			throw new GCUBEFault();
		}
	}


	@Override
	public StringMap getCapabilities(VOID request) throws RemoteException,
			GCUBEFault {
		
			ArrayList<MapItemType> toReturnList=new ArrayList<MapItemType>();
		GCUBEScope scope=ServiceContext.getContext().getScope();
		PluginManager pManager=PluginManager.get(scope);
		for(GISRepositoryType repoType:pManager.getHandledRepositories())
			for(GISDataType dataType:pManager.getHandledDataType(repoType))
				toReturnList.add(new MapItemType(repoType.toString(), dataType.toString()));
		return new StringMap(toReturnList.toArray(new MapItemType[toReturnList.size()]));		
		
	}


	@Override
	public String getPublishingReport(String request) throws RemoteException,
			GCUBEFault {
		try{
//			logger.debug("Getting report for ID "+request);
			StoredReport report=ServiceContext.getContext().getReportDao().queryForId(UUID.fromString(request));
			return report.getSerializedReport();
		} catch (Throwable t) {
			logger.error("Unexpected Exception", t);
			throw new GCUBEFault();
		}
	}
}
