/**
 * 
 */
package org.gcube.documentstore.persistence;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.gcube.common.security.AuthorizedTasks;
import org.gcube.common.security.factories.AuthorizationProvider;
import org.gcube.common.security.providers.SecretManagerProvider;
import org.gcube.common.security.secrets.Secret;
import org.gcube.documentstore.configuration.LibraryConfigurationBinder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 *
 */
class PersistenceBackendRediscover implements Runnable {
	
	private final static Logger logger = LoggerFactory.getLogger(PersistenceBackendRediscover.class);
	
	//protected final ScheduledExecutorService scheduler;
	
	protected final String context;
	protected final String initialValidContext;
	protected final FallbackPersistenceBackend fallbackPersistenceBackend;
	
	private ScheduledFuture<?> scheduledThread;
	
	public PersistenceBackendRediscover(String context, FallbackPersistenceBackend fallbackPersistenceBackend, 
			long initialDelay, long delay, TimeUnit timeUnit){
		logger.trace("Going to initialize {} for context {} with initial delay {} {} and delay {} {}", 
				PersistenceBackendRediscover.class.getSimpleName(), context, initialDelay, timeUnit, delay, timeUnit);
		this.context = context;
		try {
			Secret secret = SecretManagerProvider.get();
			this.initialValidContext = secret.getContext();
		}catch(Throwable t){
			String error = "It was not possible to obtain the initial context in " + PersistenceBackendRediscover.class.getSimpleName();
			logger.error(error, t);
			throw new RuntimeException(error, t);
		}
		logger.trace("Reference context is {} Initial context is {}", context, initialValidContext);
		this.fallbackPersistenceBackend = fallbackPersistenceBackend;
		this.scheduledThread = ExecutorUtils.PERSISTENCE_BACKEND_REDISCOVERY_POOL.scheduleAtFixedRate(this, initialDelay, delay, timeUnit);
	}

	@Override
	public void run() {
		try {
			logger.info("Trying to rediscover {} in initial context {} using generated credential", PersistenceBackend.class.getSimpleName(), initialValidContext);
			LibraryConfigurationBinder configurationBinder = LibraryConfigurationBinder.getInstance();
			AuthorizationProvider authorizationProvider = configurationBinder.getAuthorizationProvider();
			
			AuthorizedTasks.executeSafely(new Runnable() {
				@Override
				public void run() {
					PersistenceBackend rediscovered = PersistenceBackendFactory.
						rediscoverPersistenceBackend(fallbackPersistenceBackend, context);
					if(!rediscovered.getClass().equals(fallbackPersistenceBackend.getClass())){
						logger.trace("Another {} was found : {}. "
								+ "Shutting down {} Thread for context {}", 
								PersistenceBackend.class.getSimpleName(), 
								rediscovered.getClass().getSimpleName(), 
								PersistenceBackendRediscover.class.getSimpleName(),
								context);
						scheduledThread.cancel(true);
					}else{
						logger.trace("{} for context {} is still a {}. We will see if next time we will be more lucky.", 
								PersistenceBackend.class.getSimpleName(),
								FallbackPersistenceBackend.class.getSimpleName());
					}
				}
			}, authorizationProvider.getSecretForContext(initialValidContext));
		}catch(Throwable t){
			logger.info(". The attemp to rediscover {} in initial context {} failed using generated credential (Reason {})", PersistenceBackend.class.getSimpleName(), t.getMessage());
		}
	}
	
}
