package org.gcube.documentstore.records.aggregation;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.ServiceLoader;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.gcube.documentstore.exception.NotAggregatableRecordsExceptions;
import org.gcube.documentstore.persistence.ExecutorUtils;
//import org.gcube.documentstore.persistence.DefaultPersitenceExecutor;
import org.gcube.documentstore.persistence.PersistenceBackend;
import org.gcube.documentstore.persistence.PersistenceBackendConfiguration;
import org.gcube.documentstore.persistence.PersistenceExecutor;
import org.gcube.documentstore.records.AggregatedRecord;
import org.gcube.documentstore.records.Record;
import org.gcube.documentstore.records.RecordUtility;
import org.gcube.documentstore.records.implementation.ConfigurationGetPropertyValues;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 *
 */
public abstract class AggregationScheduler implements Runnable {

	public static Logger logger = LoggerFactory
			.getLogger(AggregationScheduler.class);

	protected int totalBufferedRecords;
	protected Map<String, List<Record>> bufferedRecords;

	protected final PersistenceExecutor persistenceExecutor;

	public static final int INITIAL_DELAY = 30;
	public static Integer initialDelaySet;

	public static final int DELAY = 30;
	public static Integer delaySet;

	public final static TimeUnit TIME_UNIT = TimeUnit.MINUTES;

	public static final String AGGREGATION_SCHEDULER_TIME = "AggregationSchedulerTime";

	public static final String BUFFER_RECORD_TIME = "BufferRecordTime";
	public static final String BUFFER_RECORD_NUMBER = "BufferRecordNumber";

	/**
	 * The Max amount of time for reload a configuration TODO Get from
	 * configuration
	 */
	//public static long TIME_RELOAD_CONFIGURATION = 1000 * 60 * 60 * 12; // 12
	public static long TIME_RELOAD_CONFIGURATION = 1000 * 60 * 5; // 12
																		// hour

	/**
	 * The time for first
	 */
	public static long TIME_LOAD_CONFIGURATION = 0L;

	/**
	 * Define the MAX number of Record to buffer. TODO Get from configuration
	 */
	protected static final int MAX_RECORDS_NUMBER = 100;
	protected static int maxRecordsNumberSet;

	/**
	 * The Max amount of time elapsed form last record before after that the
	 * buffered record are persisted even if TODO Get from configuration
	 */
	protected static final long OLD_RECORD_MAX_TIME_ELAPSED = 1000 * 60 * 30; // 10
																				// min
	protected static long OldRecordMaxTimeElapsedSet;

	protected ScheduledFuture<?> future = null;
	
	public static AggregationScheduler newInstance(
			PersistenceExecutor persistenceExecutor) {
		return new BufferAggregationScheduler(persistenceExecutor);
	}

	public static AggregationScheduler newInstance(
			PersistenceExecutor persistenceExecutor,
			PersistenceBackendConfiguration configuration)
			throws NumberFormatException, Exception {

		ConfigurationGetPropertyValues properties = new ConfigurationGetPropertyValues();
		Properties prop = properties.getPropValues();
		Integer delay = null;
		Integer maxRecordNumber = null;
		Integer maxRecordTime = null;

		if (prop == null) {
			// get value from service end point
			logger.trace("Configuration from service end point");
			try {
				delay = Integer.parseInt(configuration.getProperty(AGGREGATION_SCHEDULER_TIME));
			} catch (Exception e) {
				logger.trace("Configuration from service end point, not found delay value");
			}
			try {
				maxRecordTime = Integer.parseInt(configuration.getProperty(BUFFER_RECORD_TIME));
			} catch (Exception e) {
				logger.trace("Configuration from service end point, not found maxRecordTime value");
			}
			try {
				maxRecordNumber = Integer.parseInt(configuration.getProperty(BUFFER_RECORD_NUMBER));
			} catch (Exception e) {
				logger.trace("Configuration from service end point, not found maxRecordNumber value");
			}
		} else {
			// get value from properties file
			logger.trace("Configuration from properties file");
			try {
				delay = Integer.parseInt(prop.getProperty("delay"));
			} catch (Exception e) {
				logger.trace("Configuration from properties file, not found a delay value");
			}
			try {
				maxRecordNumber = Integer.parseInt(prop.getProperty("maxrecordnumber"));
			} catch (Exception e) {
				logger.trace("Configuration from properties file, not found a maxRecordNumber value");
			}
			try {
				maxRecordTime = Integer.parseInt(prop.getProperty("maxtimenumber"));
			} catch (Exception e) {
				logger.trace("Configuration from properties file, not found a maxRecordTime value");
			}

		}

		if (delay != null) {
			delaySet = delay;
			initialDelaySet = delay;
		} else {
			delaySet = DELAY;
			initialDelaySet = INITIAL_DELAY;
		}

		if (maxRecordNumber != null) {
			maxRecordsNumberSet = maxRecordNumber;
		} else {
			maxRecordsNumberSet = MAX_RECORDS_NUMBER;
		}
		if (maxRecordTime != null) {
			OldRecordMaxTimeElapsedSet = maxRecordTime * 1000 * 60;
		} else {
			OldRecordMaxTimeElapsedSet = OLD_RECORD_MAX_TIME_ELAPSED;
		}

		TIME_LOAD_CONFIGURATION = Calendar.getInstance().getTimeInMillis();
		logger.trace("Start Instance for time load configuration {}",TIME_LOAD_CONFIGURATION);

		return new BufferAggregationScheduler(persistenceExecutor);
	}

	protected AggregationScheduler(PersistenceExecutor persistenceExecutor) {
		this.bufferedRecords = new HashMap<String, List<Record>>();
		this.totalBufferedRecords = 0;
		this.persistenceExecutor = persistenceExecutor;
		if (initialDelaySet == null)
			initialDelaySet = INITIAL_DELAY;
		if (delaySet == null)
			delaySet = DELAY;
		if ((initialDelaySet == 0) || (delaySet == 0)) {
			initialDelaySet = 1;
			delaySet = 1;
		}

		scheduleTask();
		logger.trace("Thread scheduler created in {} ", this.toString());
		logger.trace("Reload configuration every {}", TIME_RELOAD_CONFIGURATION);
		logger.trace("Aggregated for max record {}", maxRecordsNumberSet);
		logger.trace("Aggregated for max time {}", OldRecordMaxTimeElapsedSet);

	}

	private void scheduleTask(){
		if (future==null){
			future = ExecutorUtils.scheduler.scheduleAtFixedRate(this, initialDelaySet, delaySet, TIME_UNIT);
		} else if (future.getDelay(TIME_UNIT)!=delaySet){
			future.cancel(false);
			future = ExecutorUtils.scheduler.scheduleAtFixedRate(this, delaySet, delaySet, TIME_UNIT);
		}
	}
	@SuppressWarnings("rawtypes")
	protected static AggregatedRecord instantiateAggregatedRecord(Record record)
			throws Exception {

		String recordType = record.getRecordType();
		Class<? extends AggregatedRecord> clz = RecordUtility
				.getAggregatedRecordClass(recordType);
		Class[] argTypes = { record.getClass() };
		Constructor<? extends AggregatedRecord> constructor = clz
				.getDeclaredConstructor(argTypes);
		Object[] arguments = { record };
		return constructor.newInstance(arguments);
	}

	@SuppressWarnings("rawtypes")
	public static AggregatedRecord getAggregatedRecord(Record record)
			throws Exception {

		AggregatedRecord aggregatedRecord;
		if (record instanceof AggregatedRecord) {
			// the record is already an aggregated version
			aggregatedRecord = (AggregatedRecord) record;
		} else {
			aggregatedRecord = instantiateAggregatedRecord(record);
		}

		return aggregatedRecord;
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	protected void madeAggregation(Record record) {

		String recordType = record.getRecordType();
		List<Record> records;

		if (this.bufferedRecords.containsKey(recordType)) {
			records = this.bufferedRecords.get(recordType);
			boolean found = false;

			for (Record bufferedRecord : records) {
				if (!(bufferedRecord instanceof AggregatedRecord)) {
					continue;
				}

				try {
					AggregatedRecord bufferedAggregatedRecord = (AggregatedRecord) bufferedRecord;

					if (record instanceof AggregatedRecord) {
						// TODO check compatibility using getAggregable
						bufferedAggregatedRecord
								.aggregate((AggregatedRecord) record);
					} else {
						bufferedAggregatedRecord.aggregate((Record) record);
					}

					logger.trace("Aggregated Record is {}",
							bufferedAggregatedRecord);
					found = true;
					break;
				} catch (NotAggregatableRecordsExceptions e) {
					logger.trace("{} is not usable for aggregation",
							bufferedRecord);
				}
			}

			if (!found) {
				try {
					records.add(getAggregatedRecord(record));
				} catch (Exception e) {
					records.add(record);
				}
				totalBufferedRecords++;
				return;
			}

		} else {
			records = new ArrayList<Record>();
			try {
				records.add(getAggregatedRecord(record));
			} catch (Exception e) {
				records.add(record);
			}
			totalBufferedRecords++;
			this.bufferedRecords.put(recordType, records);
		}

	}

	public void flush(PersistenceExecutor persistenceExecutor) throws Exception {
		aggregate(null, persistenceExecutor, true);
	}

	protected abstract void schedulerSpecificClear();

	protected void clear() {
		totalBufferedRecords = 0;
		bufferedRecords.clear();
		schedulerSpecificClear();
	}

	protected synchronized void aggregate(Record record,
			PersistenceExecutor persistenceExecutor, boolean forceFlush)
			throws Exception {

		if (record != null) {
			madeAggregation(record);
		}
		if (isTimeToPersist(maxRecordsNumberSet, OldRecordMaxTimeElapsedSet)|| forceFlush) {
			reallyFlush(persistenceExecutor);

		}
		/**
		 * reload a configuration
		 */
		long now = Calendar.getInstance().getTimeInMillis();
		if ((now - TIME_LOAD_CONFIGURATION) >= TIME_RELOAD_CONFIGURATION) {
			reloadConfiguration();
		}
	}

	protected void reloadConfiguration() throws Exception {

		new Thread() {

			public void run() {

				Integer delay = null;
				Integer maxRecordNumber = null;
				Integer maxRecordTime = null;
				try {

					ConfigurationGetPropertyValues properties = new ConfigurationGetPropertyValues();
					Properties prop = properties.getPropValues();
					if (prop != null) {
						// get value from properties file
						logger.trace("Reload Configuration from properties file");
						try {
							delay = Integer.parseInt(prop.getProperty("delay"));
						} catch (Exception e) {
							logger.trace("Reload Configuration from properties file, not found delay value");
						}
						try {
							maxRecordNumber = Integer.parseInt(prop
									.getProperty("maxrecordnumber"));
						} catch (Exception e) {
							logger.trace("Reload Configuration from properties file, not found maxRecordNumber value");
						}
						try {
							maxRecordTime = Integer.parseInt(prop
									.getProperty("maxtimenumber"));
						} catch (Exception e) {
							logger.trace("Reload Configuration from properties file, not found maxRecordTime value");
						}
					} else {
						ServiceLoader<PersistenceBackend> serviceLoader = ServiceLoader
								.load(PersistenceBackend.class);
						PersistenceBackendConfiguration configuration = null;
						for (PersistenceBackend found : serviceLoader) {
							Class<? extends PersistenceBackend> foundClass = found
									.getClass();
							try {
								String foundClassName = foundClass
										.getSimpleName();
								logger.trace("Testing {}", foundClassName);
								configuration = PersistenceBackendConfiguration
										.getInstance(foundClass);
								if (configuration == null) {
									continue;
								}
								logger.debug("{} will be used.", foundClassName);
							} catch (Exception e) {
								logger.error(
										String.format(
												"%s not initialized correctly. It will not be used. Trying the next one if any.",
												foundClass.getSimpleName()), e);
							}
						}

						if (configuration != null) {
							// get value from service end point
							logger.trace("Reload Configuration from service end point");
							try {
								delay = Integer
										.parseInt(configuration
												.getProperty(AGGREGATION_SCHEDULER_TIME));
							} catch (Exception e) {
								logger.trace("Reload Configuration from service end point,not found delay value");
							}
							try {
								maxRecordTime = Integer.parseInt(configuration
										.getProperty(BUFFER_RECORD_TIME));
							} catch (Exception e) {
								logger.trace("Reload Configuration from service end point,not found maxRecordTime value");
							}
							try {
								maxRecordNumber = Integer
										.parseInt(configuration
												.getProperty(BUFFER_RECORD_NUMBER));
							} catch (Exception e) {
								logger.trace("Reload Configuration from service end point,not found maxRecordNumber value");
							}
						}
					}

				} catch (Exception e) {
					// TODO Auto-generated catch block
					logger.error(String.format("%s not initialized correctly. It will not be used. Trying the next one if any.",e.getLocalizedMessage()), e);

				}
				// configure new value
				if (delay != null) {
					delaySet = delay;
					initialDelaySet = delay;
				} else {
					delaySet = DELAY;
					initialDelaySet = INITIAL_DELAY;
				}
				if (maxRecordNumber != null) {
					maxRecordsNumberSet = maxRecordNumber;
				} else {
					maxRecordsNumberSet = MAX_RECORDS_NUMBER;
				}
				if (maxRecordTime != null) {
					OldRecordMaxTimeElapsedSet = maxRecordTime * 1000 * 60;
				} else {
					OldRecordMaxTimeElapsedSet = OLD_RECORD_MAX_TIME_ELAPSED;
				}
				// reset a timer
				TIME_LOAD_CONFIGURATION = Calendar.getInstance()
						.getTimeInMillis();
				logger.trace("Aggregated for max record {}",
						maxRecordsNumberSet);
				logger.trace("Aggregated for max time {}",
						OldRecordMaxTimeElapsedSet);
			}

		}.start();

		
		
	}

	protected void reallyFlush(PersistenceExecutor persistenceExecutor)	throws Exception {
		if (totalBufferedRecords == 0) {
			return;
		}
		Record[] recordToPersist = new Record[totalBufferedRecords];
		int i = 0;
		Collection<List<Record>> values = bufferedRecords.values();
		for (List<Record> records : values) {
			for (Record thisRecord : records) {
				recordToPersist[i] = thisRecord;
				i++;
			}
		}
		logger.trace("reallyFlush It is time to persist buffered records {}",
				Arrays.toString(recordToPersist));
		persistenceExecutor.persist(recordToPersist);

		clear();
	}

	/**
	 * Get an usage records and try to aggregate with other buffered Usage
	 * Record.
	 * 
	 * @param singleRecord
	 *            the Usage Record To Buffer
	 * @return true if is time to persist the buffered Usage Record
	 * @throws Exception
	 *             if fails
	 */
	public void aggregate(Record record, PersistenceExecutor persistenceExecutor)
			throws Exception {
		aggregate(record, persistenceExecutor, false);
	}

	protected abstract boolean isTimeToPersist(int maxRecordNumber,
			long oldRecordMaxTime);

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
		try {
			this.flush(persistenceExecutor);
		} catch (Exception e) {
			logger.error("Error flushing Buffered Records", e);
		}
		scheduleTask();
	}

}
