package org.gcube.documentstore.records.aggregation;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.gcube.documentstore.exception.NotAggregatableRecordsExceptions;
import org.gcube.documentstore.persistence.PersistenceExecutor;
import org.gcube.documentstore.records.AggregatedRecord;
import org.gcube.documentstore.records.Record;
import org.gcube.documentstore.records.RecordUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 *
 */
public abstract class AggregationScheduler {
	
	public static Logger logger = LoggerFactory.getLogger(AggregationScheduler.class);
	
	public static AggregationScheduler newInstance(){
		return new BufferAggregationScheduler();
	}
	
	protected int totalBufferedRecords;
	protected Map<String, List<Record>> bufferedRecords;

	protected AggregationScheduler(){
		this.bufferedRecords = new HashMap<String, List<Record>>();
		this.totalBufferedRecords = 0;
	}
	
	@SuppressWarnings("rawtypes")
	protected static AggregatedRecord instantiateAggregatedRecord(Record record) throws Exception{
		String recordType = record.getRecordType();
		Class<? extends AggregatedRecord> clz = RecordUtility.getAggregatedRecordClass(recordType);
		Class[] argTypes = { record.getClass() };
		Constructor<? extends AggregatedRecord> constructor = clz.getDeclaredConstructor(argTypes);
		Object[] arguments = {record};
		return constructor.newInstance(arguments);
	}
	
	@SuppressWarnings("rawtypes")
	public static AggregatedRecord getAggregatedRecord(Record record) throws Exception {
		AggregatedRecord aggregatedRecord;
		if(record instanceof AggregatedRecord){
			// the record is already an aggregated version  
			aggregatedRecord = (AggregatedRecord) record;
		}else{
			aggregatedRecord = instantiateAggregatedRecord(record);
		}
		
		return aggregatedRecord;
	}
	
	@SuppressWarnings({ "rawtypes", "unchecked" })
	protected void madeAggregation(Record record){
		String recordType = record.getRecordType();
	
		List<Record> records;
		
		if(this.bufferedRecords.containsKey(recordType)){
			records = this.bufferedRecords.get(recordType);
			boolean found = false;
			
			for(Record bufferedRecord : records){
				if(!(bufferedRecord instanceof AggregatedRecord)){
					continue;
				}
				
				try {
					AggregatedRecord bufferedAggregatedRecord = (AggregatedRecord) bufferedRecord;
					logger.trace("Trying to use {} for aggregation.", bufferedAggregatedRecord);
					
					if(record instanceof AggregatedRecord){
						// TODO check compatibility using getAggregable
						bufferedAggregatedRecord.aggregate((AggregatedRecord) record);
					}else{
						bufferedAggregatedRecord.aggregate((Record) record);
					}
					
					logger.trace("Aggregated Record is {}", bufferedAggregatedRecord);
					found = true;
					break;
				} catch(NotAggregatableRecordsExceptions e) {
					logger.trace("{} is not usable for aggregation", bufferedRecord);
				} 
			}
			
			if(!found){
				try {
					records.add(getAggregatedRecord(record));
				} catch (Exception e) {
					records.add(record);
				}
				totalBufferedRecords++;
				return;
			}
			
			
		}else{
			records = new ArrayList<Record>();
			try {
				records.add(getAggregatedRecord(record));
			} catch (Exception e) {
				records.add(record);
			}
			totalBufferedRecords++;
			this.bufferedRecords.put(recordType, records);
		}
		
	}
	
	public void flush(PersistenceExecutor persistenceExecutor) throws Exception{
		aggregate(null, persistenceExecutor, true);
	}
	
	protected abstract void schedulerSpecificClear();
	
	protected void clear(){
		totalBufferedRecords=0;
		bufferedRecords.clear();
		schedulerSpecificClear();
	}
	
	protected synchronized void aggregate(Record record, PersistenceExecutor persistenceExecutor, boolean forceFlush) throws Exception {
		if(record!=null){
			logger.trace("Trying to aggregate {}", record);
			madeAggregation(record);
		}
		
		if(isTimeToPersist() || forceFlush){
			Record[] recordToPersist = new Record[totalBufferedRecords];
			int i = 0;
			Collection<List<Record>> values = bufferedRecords.values();
			for(List<Record> records : values){
				for(Record thisRecord: records){
					recordToPersist[i] = thisRecord;
					i++;
				}
			}
			
			logger.trace("It is time to persist buffered records {}", Arrays.toString(recordToPersist));
			persistenceExecutor.persist(recordToPersist);
			
			clear();
		}
	}
	
	
	
	/**
	 * Get an usage records and try to aggregate with other buffered
	 * Usage Record.
	 * @param singleRecord the Usage Record To Buffer
	 * @return true if is time to persist the buffered Usage Record
	 * @throws Exception if fails
	 */
	public void aggregate(Record record, PersistenceExecutor persistenceExecutor) throws Exception {
		logger.trace("Going to aggregate {}", record);
		aggregate(record, persistenceExecutor, false);
	}
	
	
	protected abstract boolean isTimeToPersist();
	
}
