/**
 * 
 */
package org.gcube.documentstore.persistence;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.gcube.documentstore.records.Record;
import org.gcube.documentstore.records.RecordUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 */
public class PersistenceBackendMonitor implements Runnable {
	
	private final static Logger logger = LoggerFactory.getLogger(PersistenceBackendMonitor.class);
	
	private final static String ELABORATION_FILE_SUFFIX = ".ELABORATION";
	private final static String ELABORATION_FILE_NOT_DELETED_SUFFIX = ".ELABORATION.NOT-DELETED";
	
	protected final ScheduledExecutorService scheduler;
	
	protected final PersistenceBackend persistenceBackend;
	
	public PersistenceBackendMonitor(PersistenceBackend persistenceBackend){
		this.persistenceBackend = persistenceBackend;
		this.scheduler = Executors.newScheduledThreadPool(1);
		this.scheduler.scheduleAtFixedRate(this, 10, 10, TimeUnit.MINUTES);
	}
	
	protected void elaborateFile(File elaborationFile){
		try(BufferedReader br = new BufferedReader(new FileReader(elaborationFile))) {
		    for(String line; (line = br.readLine()) != null; ) {
		    	try {
		    		Record record = RecordUtility.getRecord(line);
		    		persistenceBackend.accountWithFallback(record);
		    	} catch(Exception e){
		    		logger.error("Was not possible parse line {} to obtain a valid Record. Going to writing back this line as string fallback file.", line, e);
		    		FallbackPersistenceBackend fallbackPersistenceBackend = persistenceBackend.getFallbackPersistence();
		    		try {
						fallbackPersistenceBackend.printLine(line);
					} catch (Exception e1) {
						logger.error("Was not possible Line {} will be lost", line, e1);
					}
		    	}
		    }
		} catch (FileNotFoundException e) {
			logger.error("", e);
		} catch (IOException e) {
			logger.error("", e);
		}
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
		logger.debug("Trying to persist {}s which failed and were persisted using fallback", Record.class.getSimpleName());
		
		FallbackPersistenceBackend fallbackPersistenceBackend = persistenceBackend.getFallbackPersistence();
		File file = fallbackPersistenceBackend.getFallbackFile();
		File elaborationFile = null;
		synchronized (file) {
			if(file.exists()){
				elaborationFile = new File(file.getAbsolutePath()+ELABORATION_FILE_SUFFIX);
				file.renameTo(elaborationFile);
			}
		}
		
		if(elaborationFile!=null){
			synchronized (elaborationFile) {
				elaborateFile(elaborationFile);
				boolean deleted = elaborationFile.delete();
				if(!deleted){
					logger.debug("Failed to delete file {}", elaborationFile.getAbsolutePath());
					File elaborationFileNotDeleted = new File(elaborationFile.getAbsolutePath()+ELABORATION_FILE_NOT_DELETED_SUFFIX);
					elaborationFile.renameTo(elaborationFileNotDeleted);
				}
			}
			
		}
		
	}
}
