package org.gcube.data.fishfinder.tmplugin.dbconnection;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.StringTokenizer;

import org.gcube.data.fishfinder.tmplugin.FishFinderPlugin;
import org.gcube.data.fishfinder.tmplugin.repository.iterators.FishFinderRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * @author "Valentina Marioli valentina.marioli@isti.cnr.it"
 *
 */
public class FishFinderTable {

	private static Logger log = LoggerFactory.getLogger(FishFinderTable.class);
	private static final String SEPARATEDBY =  ",";
	private static final char TEXTDELIMITER = '"';
	private ArrayList<String> fields = null;
	private File tempFolder = null;

	/**
	 * Create a new table figis
	 */
	public Boolean create() {

		int[] dimension;
		String csvFile = getCsv();
		if (csvFile != null) {			
			try {
				dimension = getTableDimension(csvFile);
				createTable(csvFile, dimension);
			} catch (SQLException e) {
				return false;		
			} catch (IOException e) {
				return false;		
			}	finally{			
				if (tempFolder != null)
					clearTmp(tempFolder);
			}	
		}
		return true;		
	}


	/**
	 * Download the csv file and return the temporary url in local disk
	 */
	private String getCsv(){

		try {

			URL url = new URL("http://figisapps.fao.org/vrmf/samples/services/species/FS/extract/all.csv");
			//			URL url = new URL(FigisPlugin.baseUrl + FigisPlugin.extractAllCsv);

			tempFolder = File.createTempFile("figis-folder", "" );			
			tempFolder.delete();
			tempFolder.mkdir();	

			String csvFile = tempFolder + "/figis.csv";
			if (downloadFromUrl(url, csvFile)) {
				return csvFile;
			}
		} catch (MalformedURLException e) {
			log.error("MalformedURLException", e);
		}
		catch (IOException e) {
			log.error("IOException", e);
		}

		return null;
	}


	/**
	 * Get max table field dimension
	 */
	private int[] getTableDimension(String csvFile) throws SQLException, IOException {

		BufferedReader reader = new BufferedReader(new FileReader(csvFile));
		String line;
		int[] maxDimension = new int[0];
		int size = 0;

		while ((line = reader.readLine()) != null) {				
			ArrayList<String> tokens =  getToken(line, size);

			if (size == 0){	
				size = tokens.size();
				maxDimension = new int[size];
			}else{
				for(int i=0; i < size-1 ; i++){		
					if (tokens.get(i).length() > maxDimension[i])
						maxDimension[i] = tokens.get(i).length();					
				}				
			}
		}
		return maxDimension;
	}

	/**
	 * delete content folder
	 */
	private void clearTmp(File f) {
		if (f.exists()) {
			for (File c : f.listFiles()) {
				if (c.isDirectory())
					clearTmp(c);
				c.delete();
			}
			f.delete();
		}
		//		logger.trace("Deleted files");
	}


	/**
	 * Download csv file from URL
	 */
	private boolean downloadFromUrl(URL url, String string) throws IOException {

		boolean flag = false;
		log.info("Downloading " + string);
		InputStream is = null;
		FileOutputStream fos = null;

		try {
			URLConnection urlConn = url.openConnection();

			is = urlConn.getInputStream();
			fos = new FileOutputStream(string);

			byte[] buffer = new byte[4096];
			int len;

			while ((len = is.read(buffer)) > 0) {
				fos.write(buffer, 0, len);
			}
			flag = true;

		} finally {
			try {
				if (is != null) {
					is.close();
				}
			} finally {
				if (fos != null) {
					fos.close();
				}
			}
		}
		return flag;
	}


	/**
	 * Read a single csv line and return an ArrayList with entries in it
	 */
	private ArrayList<String> getToken(String line, int size) throws SQLException, IOException {

		ArrayList<String> tokens = new ArrayList<String>();

		Boolean complex = false;
		StringTokenizer st = null;
		//break comma separated line using ","
		st = new StringTokenizer(line, SEPARATEDBY, true);

		StringBuilder complexLine = null;
		int countDelimiter = 0;
		//		log.info("\n ");
		while(st.hasMoreTokens()){
			String token = st.nextToken();
			//			log.info("token : "+ token);
			if (token.charAt(0)==TEXTDELIMITER){
				complexLine = new StringBuilder();
				complex = true;						
			}
			if (token.charAt(token.length()-1)==TEXTDELIMITER){
				char preEnd = (token.charAt(token.length()-2));
				complexLine.append(token);
				if (preEnd == TEXTDELIMITER && token.charAt(token.length()-3)!=TEXTDELIMITER)
					continue;				
				complex = false;	
				//				log.info("complexLine : "+ complexLine);
				countDelimiter = 0;

				//remove quote at the beginning and at the end of the line
				String a = (complexLine.substring(1, complexLine.length()-1));
				String newLine = (a.toString()).replace("\"\"", "\"");
				tokens.add(newLine);
				continue;
			}

			if (complex){
				complexLine.append(token);
			}else if (token.equals(SEPARATEDBY)){
				countDelimiter++;		
				//				log.info("countDelimiter " + countDelimiter );
				if (countDelimiter > 1){
					//					log.info("put null : ");
					tokens.add("");					
				}
			}else if (!token.equals(SEPARATEDBY)){
				countDelimiter = 0;
				//				log.info("simple token : "+ token);		
				tokens.add(token);
			}

		}
		if (tokens.size() < size){
			tokens.add("");		
		}
		return tokens;
	}


	/**
	 * Create table figis by a csv file
	 */
	private boolean createTable(String csvFile, int[] dimension) throws SQLException, IOException {
		log.info("Creating table figis... ");
		ConnectionPool pool = null;
		Connection con = null;
		PreparedStatement stmtInsert = null;

		try{
			pool = ConnectionPool.getConnectionPool();
			con = pool.getConnection();

			BufferedReader reader = new BufferedReader(new FileReader(csvFile));
			String line;
			Boolean firstLine = true;
			int size = 0;
			boolean result;

			while ((line = reader.readLine()) != null) {				
				ArrayList<String> token =  getToken(line, size);				

				String query = null;
				if (firstLine){
					size = token.size();
					this.fields = token;
					//					System.out.println(this.fields);
					query = createTableQuery(dimension);
					firstLine = false;		
					result = pool.insertPreStatement(query);
					if (!result)
						log.error("problem creating table");
					//					log.trace(fields.toString());
				}else{
					//insert line in table				
					query = createInsertQuery(token);	
					if (!pool.preStatement(query, token, stmtInsert))
						log.error("error ");					
				}
			}

		} catch (ConnectionPoolException e) {
			log.error("ConnectionPoolException", e);
		} finally {	
			if ((pool!=null) && (con!=null)){
				pool.releaseConnection(con);
			}
		}
		return true;

	}

	/**
	 * Create query to insert entries
	 */
	private String createInsertQuery(ArrayList<String> token) {


		StringBuilder query = new StringBuilder();
		Boolean first = true;
		query.append("insert into " + FishFinderPlugin.table + " (");

		for (String field: fields){
			//						log.info("entry " + field);
			if (!first)
				query.append(", ");
			else
				first = false;
			query.append(field);		
		}
		query.append(")");
		query.append(" values (");

		first = true;

		for (int i=0; i< token.size(); i++){
			if (!first)
				query.append(", ");
			else
				first = false;
			query.append("?");			
		}
		query.append(")");

		return (query.toString());
	}

	/**
	 * Create query to create table by csv header
	 */
	private String createTableQuery(int[] dimension) {
		Boolean firstElement = true;
		StringBuilder query = new StringBuilder();
		query.append("create table " + FishFinderPlugin.table + " (");
		int i = 0;
		for (String field: fields){

			//length for type varchar must be at least 1
			if (dimension[i] <1)
				dimension[i] = 1;

			if (firstElement){
				String newField = field.replace("3", "three");
				this.fields.set(i, newField);
				//				this.newFields.add(newField);
				query.append(newField);
				query.append(" varchar(" + dimension[i] + ") NOT NULL PRIMARY KEY");
				firstElement = false;
			}
			else{
				//				this.newFields.add(field);
				query.append(", ");
				query.append(field);
				query.append(" varchar(" + dimension[i] + ")");
			}

			i++;
		}
		query.append(")");

		return (query.toString());
	}


	/**
	 * get a result set by a scientific name
	 */
	public ResultSet getAllRecords() {
		ConnectionPool pool = null;
		Connection con = null;
		ResultSet results = null;

		try {
			pool = ConnectionPool.getConnectionPool();			
			con = pool.getConnection();
			String query = "select * from figis";	
			results =  pool.selectPreStatement(query);
		}
		catch (Throwable e) {
			log.error("general Error", e);
		}finally{
			if ((pool!=null) && (con!=null)){
				pool.releaseConnection(con);
			}			
		}
		return results;
	}




	/**
	 * get a result set by a scientific name
	 */
	public FishFinderRecord getRecordByID(String id) {
		log.info("getRecordByID " + id);
		ConnectionPool pool = null;
		Connection con = null;
		ResultSet result = null;
		FishFinderRecord record = null;

		try {
			pool = ConnectionPool.getConnectionPool();
			con = pool.getConnection();

			String three_alpha_code = "%" + id + "%";
			String query = "select * from figis where UPPER(three_alpha_code) like UPPER(?)";	

			result =  pool.selectPrestatement(query, three_alpha_code);

			if (result.next())
				record = new FishFinderRecord(result);

		}
		catch (Throwable e) {
			log.error("general Error", e);
		}finally{
			if ((pool!=null) && (con!=null)){
				pool.releaseConnection(con);
			}	
			try {
				if (result != null) {
					result.close();
				}
			} catch (SQLException ex) {
				log.error("sql Error", ex);
			}
		}
		return record;
	}


	public boolean createTabUpdates() throws SQLException {
		// create table updates (id serial NOT NULL PRIMARY KEY, date date);
		log.info("Creating table figis... ");
		ConnectionPool pool = null;
		Connection con = null;

		try{
			pool = ConnectionPool.getConnectionPool();
			con = pool.getConnection();
			boolean result;

			String query = "create table updates (id serial NOT NULL PRIMARY KEY, date date)";
			result = pool.insertPreStatement(query);
			if (!result)
				log.error("problem creating table");

		} catch (ConnectionPoolException e) {
			log.error("ConnectionPoolException", e);
		} finally {	
			if ((pool!=null) && (con!=null)){
				pool.releaseConnection(con);
			}
		}
		return true;

	}
}
