package org.gcube.data.analysis.tabulardata.query;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.enterprise.inject.Default;
import javax.inject.Inject;

import org.gcube.common.database.endpoint.DatabaseEndpoint;
import org.gcube.common.database.endpoint.DatabaseProperty;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.exceptions.NoSuchTableException;
import org.gcube.data.analysis.tabulardata.cube.exceptions.TableCreationException;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableCreator;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.metadata.levels.Level;
import org.gcube.data.analysis.tabulardata.model.metadata.levels.Levels;
import org.gcube.data.analysis.tabulardata.model.relationship.TableRelationship;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableType;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

@Default
public class TabularQuery {

	Logger log = LoggerFactory.getLogger(TabularQuery.class);

	CubeManager cm;

	@Inject
	public TabularQuery(CubeManager cm) {
		this.cm = cm;
	}

	/**
	 * Get the size (in terms of tuples) of a table
	 * 
	 * @param tableId
	 *            the table id
	 */
	public int getSize(long tableId) {

		// Retrieve tablename from metadata
		String tableName;
		try {
			tableName = cm.getTable(tableId).getName();
		} catch (NoSuchTableException e) {
			log.error(String.format("Unable to retrieve table metadata for table with id %s", tableId), e);
			throw new RuntimeException("Unable to retrieve metadata for the table. Check server logs");
		}

		// Obtain row count
		String sql = String.format("SELECT COUNT(*) FROM $1%s;", tableName);
		ResultSet rs = executeQuery(sql);
		try {
			rs.next();
			return rs.getInt(1);
		} catch (SQLException e) {
			log.error(String.format("An error occurred while querying the database.", tableId), e);
			throw new RuntimeException("An error occurred while querying the database. Check server logs");
		}

	}

	/**
	 * Query a table for tuples
	 * 
	 * @param tableId
	 *            the table id
	 * @param offset
	 * @param pagesize
	 * @param orderingColumnName
	 * @param order
	 */
	public Iterator<String[]> query(long tableId, int offset, int pagesize, String orderingColumnName, Order order) {

		Table t = null;

		// retrieve table metadata
		try {
			t = cm.getTable(tableId);
		} catch (NoSuchTableException e) {
			log.error("Error occurred while querying table", e);
			throw new RuntimeException(String.format(
					"Unable to retrieve metadata for the given table id '%s' . Check server logs.", tableId));
		}

		// Check if provided ordering column is valid
		try {
			t.getColumnByName(orderingColumnName);
		} catch (Exception e) {
			log.error(String.format("Provided ordering column name '%s' is not valid within table %s.",
					orderingColumnName, t));
			throw new IllegalArgumentException("Given ordering column with name " + orderingColumnName
					+ " does not exists within the table with id " + tableId);
		}

		// Query building

		// Collect column names
		String columnNames = "";
		Collection<Column> columns = t.getColumns();
		int i = 1;
		for (Column c : columns) {
			columnNames += c.getName();
			if (i++ < columns.size())
				columnNames += ",";
		}

		//
		String orderString = null;
		if (order == Order.ASCENDING)
			orderString = "ASC";
		else
			orderString = "DESC";

		String sql = String.format("SELECT %1$s FROM %2$s ORDER BY %3$s %4$s LIMIT %5$s OFFSET %6$s;", columnNames,
				t.getName(), orderingColumnName, orderString, pagesize, offset);

		return new SQLResultSetIterator(executeQuery(sql));

	}

	public Iterator<String[]> query(long tableId, int offset, int pagesize) {

		Table t = null;

		try {
			t = cm.getTable(tableId);
		} catch (NoSuchTableException e) {
			log.error("Error occurred while querying table", e);
			throw new RuntimeException(String.format(
					"Unable to retrieve metadata for the given table id '%s'. Check server logs.", tableId));
		}

		String columnNames = "";
		Collection<Column> columns = t.getColumns();
		int i = 1;
		for (Column c : columns) {
			columnNames += c.getName();
			if (i++ < columns.size())
				columnNames += ",";
		}
		String sql = String.format("SELECT %s FROM %s LIMIT %s OFFSET %s;", columnNames, t.getName(), pagesize, offset);
		log.debug("Executing sql statement: " + sql);

		return new SQLResultSetIterator(executeQuery(sql));

	}

	public String queryAsJSON(long tableId, int offset, int pagesize) {
		Iterator<String[]> iterator = query(tableId, offset, pagesize);
		Table t = getTable(tableId);
		return generateJSON(t.getColumns(), iterator);
	}

	public String queryAsJSON(long tableId, int offset, int pagesize, String orderingColumnName, Order order) {
		Iterator<String[]> iterator = query(tableId, offset, pagesize);
		Table t = getTable(tableId);
		return generateJSON(t.getColumns(), iterator);
	}

	/**
	 * Generate a dataset table view with level view column
	 * 
	 * @param tableId
	 *            the id of the dataset table
	 * @return A table representing the generated materialized view table
	 */
	public Table generateDatasetView(long tableId) {
		Table dataset = null;
		try {
			dataset = cm.getTable(tableId);
		} catch (NoSuchTableException e) {
			log.error("Cannot find a table with the id: " + tableId + ".");
			throw new IllegalArgumentException("A table with the given id does not exists.", e);
		}
		if (dataset.getTableType() != TableType.DATASET)
			throw new RuntimeException("The table with the given id is not a dataset.");
		TableCreator ttc = cm.createTable(TableType.VIEWTABLE);
		try {
			ttc.like(dataset, false);
			Collection<Column> crCols = dataset.getColumns(ColumnType.CODELISTREF);
			for (Column column : crCols) {
				Table refTable = cm.getTable(column.getRelationship().getTargetTableRef().getTableId());
				List<Level> levels = refTable.getMetadataObject(Levels.class);
				int i = 0;
				for (String viewColumnName : levels.get(0).getViewColumnNames()) {
					Column refColumn = refTable.getColumnByName(viewColumnName);
					refColumn.setName(column.getName() + "_view" + i);
					refColumn.setLabel(column.getName() + "_view" + i);
					i++;
					ttc.addColumn(refColumn);
				}
			}
			Table viewTable = ttc.create();
			log.debug("Created view: " + viewTable);
			// Fill with data

			// Create SELECT statement
			Collection<Table> refTables = Lists.newArrayList();
			for (TableRelationship rel : dataset.getRelationships()) {
				refTables.add(cm.getTable(rel.getTargetTableRef().getTableId()));
			}
			StringBuilder sqlSelect = new StringBuilder();
			sqlSelect.append("SELECT ");
			for (Column column : dataset.getColumns()) {
				if (column.getColumnType() == ColumnType.ID)
					continue;
				sqlSelect.append("d." + column.getName() + " ,");
			}
			int i = 0;
			for (Table table : refTables) {
				List<Level> levels = table.getMetadataObject(Levels.class);
				// TODO for now handle single level only. Multilevel management
				// with hcl and such will be done in the future
				Level level = levels.get(0);
				for (String levelColumnName : level.getViewColumnNames()) {
					sqlSelect.append("c" + i + "." + levelColumnName + " ,");
				}
				i++;
			}
			sqlSelect.deleteCharAt(sqlSelect.length() - 1); // Remove last comma
			sqlSelect.append("FROM " + dataset.getName() + " AS d ");
			i = 0;
			for (TableRelationship rel : dataset.getRelationships()) {

				Table refTable = cm.getTable(rel.getTargetTableRef().getTableId());
				String refTableAlias = "c" + i;
				sqlSelect.append(" LEFT JOIN " + refTable.getName() + " AS " + refTableAlias + " ON d."
						+ rel.getSourceTableFKRef().getColumnName() + " = " + refTableAlias + ".id");
				i++;
			}
			// Create INSERT statement
			StringBuilder orderedColumnNames = new StringBuilder();
			for (Column column : viewTable.getColumns()) {
				orderedColumnNames.append(column.getName() + " ,");
			}
			orderedColumnNames.deleteCharAt(orderedColumnNames.length() - 1); // Remove
																				// last
																				// comma
			String sqlInsert = String.format("INSERT INTO %s (%s) ", viewTable.getName(), orderedColumnNames);

			String sql = sqlInsert + sqlSelect + ";";
			log.debug("Executing view filling SQL statement: " + sql);
			execute(sql);

			return viewTable;
		} catch (TableCreationException e) {
			log.error("Unable to create table", e);
			throw new RuntimeException("An error occurred during table creation. Check server logs.");
		} catch (NoSuchTableException e) {
			log.error("Unable to find referenced table.", e);
			throw new RuntimeException("An error occurred during table creation. Check server logs.");
		} catch (Exception e) {
			log.error("Generic exception detected.", e);
			throw new RuntimeException("An error occurred during table creation. Check server logs.");
		}
	}

	private String generateJSON(List<Column> columns, Iterator<String[]> dataIterator) {
		JSONObject json = new JSONObject();
		JSONArray jsonColHeaders = new JSONArray();
		for (Column c : columns) {
			jsonColHeaders.put(c.getName());
		}
		JSONArray jsonRows = new JSONArray();
		while (dataIterator.hasNext()) {
			String[] row = dataIterator.next();
			jsonRows.put(row);
		}
		try {
			json.put("headers", jsonColHeaders);
			json.put("rows", jsonRows);
		} catch (JSONException e) {
			log.error("Unable to produce JSON document.", e);
			throw new RuntimeException("Error occured with serialization of table content. Check server log.");
		}
		return json.toString();
	}

	private Table getTable(long tableId) {
		try {
			return cm.getTable(tableId);
		} catch (NoSuchTableException e) {
			log.error(String.format("A table with id '%s' does not exists.", tableId));
			throw new IllegalArgumentException(String.format("A table with id '%s' does not exists", e));
		}
	}

	private Connection getConnection() {
		DatabaseEndpoint databaseEndpoint = cm.getUnprivilegedDatabaseEndpoint();
		String driver = null;
		for (DatabaseProperty p : databaseEndpoint.getProperties()) {
			if (p.getKey().equals("driver"))
				driver = p.getValue();
		}

		if (driver == null | driver.isEmpty())
			throw new RuntimeException("Unable to locate driver to use for the DB connection.");

		try {

			Class.forName(driver);

		} catch (ClassNotFoundException e) {
			System.out.println("Where is your " + driver + " JDBC Driver? " + "Include in your library path!");
			e.printStackTrace();
			throw new RuntimeException("Unable to find DB driver: " + driver);
		}

		try {
			return DriverManager.getConnection(databaseEndpoint.getConnectionString(), databaseEndpoint
					.getCredentials().getUsername(), databaseEndpoint.getCredentials().getPassword());
		} catch (SQLException e) {
			System.out.println("Connection Failed! Check output console");
			e.printStackTrace();
			throw new RuntimeException("Unable to estabilish a connection to the db using connection string: "
					+ databaseEndpoint.getConnectionString());
		}
	}

	private ResultSet executeQuery(String sql) {
		Connection conn = getConnection();
		Statement s = null;
		try {
			s = conn.createStatement();
			ResultSet rs = s.executeQuery(sql);
			conn.close();
			return rs;
		} catch (SQLException e) {
			log.error("Unable to execute SQL query.", e);
			throw new RuntimeException("Unable to query the DB. Check server logs.");
		}
	}

	private void execute(String sql) {
		Connection conn = getConnection();
		Statement s = null;
		try {
			s = conn.createStatement();
			s.execute(sql);
			conn.close();
		} catch (SQLException e) {
			log.error("Unable to execute SQL query.", e);
			throw new RuntimeException("Unable to query the DB. Check server logs.");
		}
	}

}
