package org.gcube.data.analysis.tabulardata.model.table;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;

import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.idioms.ColumnHasLabel;
import org.gcube.data.analysis.tabulardata.model.idioms.ColumnHasName;
import org.gcube.data.analysis.tabulardata.model.idioms.ColumnIsOfType;
import org.gcube.data.analysis.tabulardata.model.metadata.CubeMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.HashMapMetadataHolder;
import org.gcube.data.analysis.tabulardata.model.metadata.MetadataHolder;
import org.gcube.data.analysis.tabulardata.model.reference.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.reference.ColumnReferenceImpl;
import org.gcube.data.analysis.tabulardata.model.reference.TableReference;
import org.gcube.data.analysis.tabulardata.model.reference.TableReferenceImpl;
import org.gcube.data.analysis.tabulardata.model.relationship.TableRelationship;
import org.gcube.data.analysis.tabulardata.model.relationship.TableRelationshipImpl;

import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;

/**
 * Represent a generic relational table.
 * 
 * @author "Luigi Fortunati"
 * 
 */
public abstract class Table implements MetadataHolder {

	/**
	 * The unique id of the table
	 */
	private Long id = null;

	/**
	 * The relational table name
	 */
	private String name = null;

	private MetadataHolder delegate = new HashMapMetadataHolder();

	private Long parentTableId = null;

	private boolean exists = true;

	/**
	 * @param id
	 *            The unique id of the table
	 * @param name
	 *            the relational table name
	 */
	public Table(long id, String name) {
		super();
		if (name == null || name.isEmpty())
			throw new IllegalArgumentException("Table name cannot be null or empty");
		this.id = id;
		this.name = name;
	}

	public Table(String name) {
		if (name == null || name.isEmpty())
			throw new IllegalArgumentException("Table name cannot be null or empty");
		this.name = name;
	}

	/**
	 * @return the unique id of the relational table
	 */
	public Long getId() {
		return id;
	}

	/**
	 * @return the database name of the relational table
	 */
	public String getName() {
		return name;
	}

	public TableReference getReference() {
		return new TableReferenceImpl(getId());
	}
	
	public ColumnReference getColumnReference(Column column){
		if (!getColumns().contains(column)) throw new IllegalArgumentException(String.format("Column %1$s is not contained in table %2$s. Unable to produce a reference for the column.", column, this)); 
			return new ColumnReferenceImpl(getReference(), column.getName());
	}

	/**
	 * @return the table type enumarated value
	 */
	public abstract TableType getTableType();

	/**
	 * @return the column set of the table
	 */
	public abstract List<Column> getColumns();

	public Column getColumnByName(String columnName) throws RuntimeException, IllegalArgumentException {
		Collection<Column> columns = Collections2.filter(getColumns(), new ColumnHasName(columnName));
		if (columns.size() > 1)
			throw new RuntimeException("Found multiple column with the same name '" + columnName + "'.");
		if (columns.size() == 0)
			throw new IllegalArgumentException("No column found with name '" + columnName + "'.");
		return columns.iterator().next();
	}

	public Column getColumnByLabel(String columnLabel) throws RuntimeException, IllegalArgumentException {
		Collection<Column> columns = Collections2.filter(getColumns(), new ColumnHasLabel(columnLabel));
		if (columns.size() > 1)
			throw new RuntimeException("Found multiple column with the same label '" + columnLabel + "'.");
		if (columns.size() == 0)
			throw new IllegalArgumentException("No column found with label '" + columnLabel + "'.");
		return columns.iterator().next();
	}

	public List<Column> getColumns(ColumnType columnType) {
		return Lists.newArrayList(Collections2.filter(getColumns(), new ColumnIsOfType(columnType)));
	}

	public List<Column> getColumns(ColumnType... columnTypes) {
		List<Column> result = Lists.newArrayList();
		for (ColumnType columnType : columnTypes) {
			result.addAll(Collections2.filter(getColumns(), new ColumnIsOfType(columnType)));
		}
		return result;
	}

	public boolean hasRelationships() {
		for (Column column : getColumns()) {
			if (column.hasRelationship())
				return true;
		}
		return false;
	}

	public Collection<TableRelationship> getRelationships() {
		Collection<TableRelationship> result = Lists.newArrayList();
		for (Column column : getColumns()) {
			if (column.hasRelationship())
				result.add(new TableRelationshipImpl(new ColumnReferenceImpl(getId(), column.getName()), column
						.getRelationship()));
		}
		return result;
	}

	public boolean sameStructureAs(Table table) {
		if (!this.name.equals(table.getName()))
			return false;
		if (!this.getTableType().equals(table.getTableType()))
			return false;
		for (Column column : getColumns())
			if (!table.getColumns().contains(column))
				return false;
		for (Column column : table.getColumns())
			if (!getColumns().contains(column))
				return false;
		return true;
	}

	

	public Long getParentTableId() {
		return parentTableId;
	}

	public void setParentTableId(Long parentTableId) {
		this.parentTableId = parentTableId;
	}

	

	public boolean exists() {
		return exists;
	}

	public void setExists(boolean exists) {
		this.exists = exists;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((delegate == null) ? 0 : delegate.hashCode());
		result = prime * result + (exists ? 1231 : 1237);
		result = prime * result + ((id == null) ? 0 : id.hashCode());
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		result = prime * result + ((parentTableId == null) ? 0 : parentTableId.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Table other = (Table) obj;
		if (delegate == null) {
			if (other.delegate != null)
				return false;
		} else if (!delegate.equals(other.delegate))
			return false;
		if (exists != other.exists)
			return false;
		if (id == null) {
			if (other.id != null)
				return false;
		} else if (!id.equals(other.id))
			return false;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		if (parentTableId == null) {
			if (other.parentTableId != null)
				return false;
		} else if (!parentTableId.equals(other.parentTableId))
			return false;
		return true;
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("Table [getId()=");
		builder.append(getId());
		builder.append(", getTableType()=");
		builder.append(getTableType());
		builder.append(", getName()=");
		builder.append(getName());
		builder.append(", exists()=");
		builder.append(exists());
		builder.append(", getParentTableId()=");
		builder.append(getParentTableId());
		builder.append(", getAllMetadata()=");
		builder.append(getAllMetadata());
		builder.append(", getColumns()=\n");
		for (Column column : getColumns()) {
			builder.append("\t"+ column +"\n");
		}
		builder.append(", getRelationships()=");
		builder.append(getRelationships());
		builder.append("]");
		return builder.toString();
	}

	public <C extends CubeMetadata<? extends Serializable>> C getMetadata(Class<C> metadataType) {
		return delegate.getMetadata(metadataType);
	}

	public void removeMetadata(Class<? extends CubeMetadata<? extends Serializable>> metadataType) {
		delegate.removeMetadata(metadataType);
	}

	public void setMetadata(CubeMetadata<? extends Serializable> metadata) {
		delegate.setMetadata(metadata);
	}

	public Collection<CubeMetadata<Serializable>> getAllMetadata() {
		return delegate.getAllMetadata();
	}

	public void removeAllMetadata() {
		delegate.removeAllMetadata();
	}

	public void setMetadata(Collection<? extends CubeMetadata<? extends Serializable>> metadataObjects) {
		delegate.setMetadata(metadataObjects);
	}

	public <T extends Serializable> T getMetadataObject(Class<? extends CubeMetadata<T>> metadataType) {
		return delegate.getMetadataObject(metadataType);
	}

	

}