package org.gcube.data.analysis.tabulardata.service.impl.tabular;

import static org.gcube.data.analysis.tabulardata.clientlibrary.plugin.AbstractPlugin.history;

import java.io.StringReader;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.common.homelibrary.home.workspace.folder.items.GCubeItem;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.data.analysis.tabulardata.clientlibrary.proxy.HistoryManagerProxy;
import org.gcube.data.analysis.tabulardata.metadata.NoSuchMetadataException;
import org.gcube.data.analysis.tabulardata.service.exception.NoSuchTabularResourceException;
import org.gcube.data.analysis.tabulardata.service.impl.HLUtils;
import org.gcube.data.analysis.tabulardata.service.tabular.HistoryStep;
import org.gcube.data.analysis.tabulardata.service.tabular.TabularResource;
import org.gcube.data.analysis.tabulardata.service.tabular.TabularResourceId;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.DescriptionMetadata;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.NameMetadata;
import org.gcube.data.analysis.tabulardata.service.tabular.metadata.TabularResourceMetadata;
import org.gcube.data.td.commons.webservice.types.HistoryData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;
import com.thoughtworks.xstream.XStream;

public class TabularResourceObject implements TabularResource{

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private static Logger logger = LoggerFactory.getLogger(TabularResourceObject.class);

	private HistoryManagerProxy historyManager = history().build();

	private static final XStream xstream = new XStream();
	private static final String ITEM_METADATA = "metadata";
	private static final String TABULAR_RESOURCE_ID = "taularResourceId";

	public static List<TabularResource> getAll(String user){
		try{
			List<? extends WorkspaceItem> children = HLUtils.getAppFolder(user).getChildren();

			List<TabularResource> toReturn = Lists.newArrayList();

			for (WorkspaceItem item : children)
				if (item instanceof GCubeItem){
					GCubeItem gcubeItem = ((GCubeItem)item);
					TabularResourceObject trObj = new TabularResourceObject(gcubeItem);
					toReturn.add(trObj);
				}

			return toReturn;

		}catch (Exception e) {
			logger.error("error retrieving tabular resources ",e);
			throw new RuntimeException(e);
		}
	}

	public static TabularResource get(String user, TabularResourceId id) throws NoSuchTabularResourceException{
		try{
			List<? extends WorkspaceItem> children = HLUtils.getAppFolder(user).getChildren();

			for (WorkspaceItem item : children)
				if (item instanceof GCubeItem){
					GCubeItem gcubeItem = ((GCubeItem)item);
					TabularResourceObject trObj = new TabularResourceObject(gcubeItem);
					if (trObj.getId().equals(id))
						return trObj;
				}

		}catch (Exception e) {
			logger.error("error retrieving tabular resources ",e);
			throw new RuntimeException(e);
		}

		throw new NoSuchTabularResourceException(id);
	}

	public static TabularResource create(String user, TabularResourceId id) throws Exception{
		Map<String, String> properties = new HashMap<String, String>();
		properties.put(TABULAR_RESOURCE_ID, id.getValue()+"");
		GCubeItem item = (GCubeItem) HLUtils.getWorkspace(user).createGcubeItem("tabularResource_"+id.getValue(), "a tabular resource",
				Collections.singletonList(ScopeProvider.instance.get()), user, "TabularResource", properties, HLUtils.getAppFolder(user).getId());
		System.out.println("item id is "+item.getId() );
		return new TabularResourceObject(item);
	}
	
	
	private String gcubeItemId;

	private TabularResourceId id;

	private MetadataList metadata;

	private String name;
	private String description;
	private String user;

	public TabularResourceObject(GCubeItem item) throws InternalErrorException{
		this.gcubeItemId = item.getId();
		if (item.getItemProperties().containsKey(ITEM_METADATA))
			this.metadata = (MetadataList) xstream.fromXML(new StringReader(item.getItemProperties().get(ITEM_METADATA)));
		else this.metadata = new MetadataList();
		this.id = new TabularResourceId(Long.parseLong(item.getItemProperties().get(TABULAR_RESOURCE_ID)));
		this.name = item.getName();
		this.description = item.getDescription();
		this.metadata.add(new NameMetadata(this.name));
		this.metadata.add(new DescriptionMetadata(this.description));
		user = item.getOwner().getPortalLogin();
	}

	@Override
	public <C extends TabularResourceMetadata> C getMetadata(
			Class<C> metadataType) throws NoSuchMetadataException {
		return this.metadata.get(metadataType);
	}

	@Override
	public void removeMetadata(
			Class<? extends TabularResourceMetadata> metadataType) {
		this.metadata.remove(metadataType);
		flush();
	}

	@Override
	public void setMetadata(TabularResourceMetadata metadata) {
		checkNameAndDescription(metadata);
		this.metadata.add(metadata);
		flush();

	}

	@Override
	public Collection<TabularResourceMetadata> getAllMetadata() {
		return this.metadata.getAll();
	}

	@Override
	public void setAllMetadata(Collection<TabularResourceMetadata> metadata) {
		for (TabularResourceMetadata singleMetadata: metadata ){
			checkNameAndDescription(singleMetadata);
			this.metadata.add(singleMetadata);
		}
		flush();
	}

	@Override
	public void removeAllMetadata() {
		this.metadata.reset();
		flush();
	}

	@Override
	public TabularResourceId getId() {
		return id;
	}

	@Override
	public List<HistoryStep> getHistory() {
		try{
			List<HistoryStep> historySteps = Lists.newArrayList();
			List<HistoryData> steps = historyManager.getHistory(id.getValue());
			for (HistoryData step: steps)
				historySteps.add(new HistoryStepImpl(step));
			return historySteps;
		}catch (Exception e) {
			logger.error("error retrievin history for tabular resource id {}",id);
			throw new RuntimeException(e);
		}
	}

	private void checkNameAndDescription(TabularResourceMetadata metadata){
		if(metadata instanceof NameMetadata){
			this.name = ((NameMetadata)metadata).getValue();
			try {
				HLUtils.getWorkspace(user).renameItem(this.gcubeItemId, this.name);
			} catch (Exception e) {
				new RuntimeException(e);
			}
		}
		if(metadata instanceof DescriptionMetadata)
			this.description = ((DescriptionMetadata)metadata).getValue();
	}
	
	private void flush(){
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "TabularResourceObject [historyManager=" + historyManager
				+ ", gcubeItemId=" + gcubeItemId + ", id=" + id + ", metadata="
				+ metadata + ", name=" + name + ", description=" + description
				+ ", user=" + user + "]";
	}
	
	
}
