package org.gcube.data.analysis.tabulardata.statistical;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.gcube.common.homelibrary.home.Home;
import org.gcube.common.homelibrary.home.workspace.Workspace;
import org.gcube.common.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.common.homelibrary.home.workspace.folder.items.ExternalImage;
import org.gcube.data.analysis.statisticalmanager.proxies.StatisticalManagerDataSpace;
import org.gcube.data.analysis.statisticalmanager.proxies.StatisticalManagerFactory;
import org.gcube.data.analysis.statisticalmanager.stubs.types.SMComputationConfig;
import org.gcube.data.analysis.statisticalmanager.stubs.types.SMComputationRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.types.SMOperationStatus;
import org.gcube.data.analysis.statisticalmanager.stubs.types.SMResourceType;
import org.gcube.data.analysis.statisticalmanager.stubs.types.schema.SMAbstractResource;
import org.gcube.data.analysis.statisticalmanager.stubs.types.schema.SMComputation;
import org.gcube.data.analysis.statisticalmanager.stubs.types.schema.SMEntries;
import org.gcube.data.analysis.statisticalmanager.stubs.types.schema.SMFile;
import org.gcube.data.analysis.statisticalmanager.stubs.types.schema.SMInputEntry;
import org.gcube.data.analysis.statisticalmanager.stubs.types.schema.SMObject;
import org.gcube.data.analysis.statisticalmanager.stubs.types.schema.SMOperationInfo;
import org.gcube.data.analysis.statisticalmanager.stubs.types.schema.SMResource;
import org.gcube.data.analysis.statisticalmanager.stubs.types.schema.SMTable;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.metadata.NoSuchMetadataException;
import org.gcube.data.analysis.tabulardata.model.metadata.table.GenericMapMetadata;
import org.gcube.data.analysis.tabulardata.model.resources.InternalURI;
import org.gcube.data.analysis.tabulardata.model.resources.ResourceType;
import org.gcube.data.analysis.tabulardata.model.resources.TableResource;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerStatus;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerWrapper;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.data.analysis.tabulardata.operation.worker.results.ResourcesResult;
import org.gcube.data.analysis.tabulardata.operation.worker.results.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.ImmutableTableResource;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.ImmutableURIResult;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.ResourceDescriptorResult;
import org.gcube.data.analysis.tabulardata.operation.worker.types.DataWorker;
import org.gcube.data.analysis.tabulardata.operation.worker.types.ResourceCreatorWorker;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.PrimitiveTypes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class StatisticalOperation extends ResourceCreatorWorker {

	private static Logger logger = LoggerFactory.getLogger(StatisticalOperation.class);
	
	private StatisticalManagerDataSpace statisticalDataSpace;
	private StatisticalManagerFactory statisticalManagerFactory;
	private ExportToStatisticalOperationFactory exportFactory;
	private ImportFromStatisticalOperationFactory importFactory;
	private Home home;

	private CubeManager cubeManager;

	public StatisticalOperation(OperationInvocation sourceInvocation,
			StatisticalManagerDataSpace statisticalDataSpace,
			StatisticalManagerFactory statisticalManagerFactory,
			ExportToStatisticalOperationFactory exportFactory,
			ImportFromStatisticalOperationFactory importFactory, Home home, CubeManager cubeManager) {
		super(sourceInvocation);
		this.statisticalDataSpace = statisticalDataSpace;
		this.statisticalManagerFactory = statisticalManagerFactory;
		this.exportFactory = exportFactory;
		this.importFactory = importFactory;
		this.home=home;
		this.cubeManager = cubeManager;

	}

	// Invocation parameters
	private Map<String, Object> algorithmParameters;
	private String user;
	private String algorithmId;
	private String experimentTitle = null;
	private String experimentDescription = null;
	private Table targetTable;

	private String dataSpaceTableId;


	private Table resultTable;



	//	private ArrayList<Table> collaterals=new ArrayList<Table>();

	private List<ResourceDescriptorResult> results=new ArrayList<ResourceDescriptorResult>();

	// Execution

	String computationId;

	@Override
	protected ResourcesResult execute() throws WorkerException {
		getParameters();
		updateProgress(0.05f,"Trasnferring to data space");		
		importIntoDataSpace();
		updateProgress(0.2f,"Submitting computation");
		submitComputation();
		waitForComputation();
		generateTableFromResult();
		return new ResourcesResult(results);
	}

	public void getParameters() {
		Map<String, Object> params = getSourceInvocation()
				.getParameterInstances();
		algorithmParameters = (Map<String, Object>) params
				.get(StatisticalOperationFactory.SM_ENTRIES.getIdentifier());
		user = (String) params.get(StatisticalOperationFactory.USER
				.getIdentifier());
		algorithmId = (String) params.get(StatisticalOperationFactory.ALGORITHM
				.getIdentifier());
		if (params.containsKey(StatisticalOperationFactory.TITLE
				.getIdentifier()))
			experimentTitle = (String) params
			.get(StatisticalOperationFactory.TITLE.getIdentifier());
		if (params.containsKey(StatisticalOperationFactory.DESCRIPTION
				.getIdentifier()))
			experimentDescription = (String) params
			.get(StatisticalOperationFactory.DESCRIPTION
					.getIdentifier());
		targetTable = cubeManager.getTable(getSourceInvocation()
				.getTargetTableId());
	}

	public void submitComputation() {
		SMComputationRequest request = new SMComputationRequest();
		SMComputationConfig config = new SMComputationConfig();

		List<SMInputEntry> entryList = new ArrayList<SMInputEntry>();
		for (Entry<String, Object> mapEntry : algorithmParameters.entrySet())
			if (((String) mapEntry.getValue()).equals(targetTable.getId()
					.toString()))
				entryList.add(new SMInputEntry(mapEntry.getKey(),
						dataSpaceTableId));
			else
				entryList.add(new SMInputEntry(mapEntry.getKey(), mapEntry
						.getValue().toString()));

		config.parameters(new SMEntries(entryList
				.toArray(new SMInputEntry[entryList.size()])));

		config.algorithm(algorithmId);
		request.user(user);
		request.title((experimentTitle == null ? algorithmId
				: experimentTitle));
		request.description((experimentDescription == null ? "Submission via TDM"
				: experimentDescription));
		request.config(config);
		computationId = statisticalManagerFactory.executeComputation(request);
	}

	public void waitForComputation() throws WorkerException {
		boolean complete = false;
		while (!complete) {
			SMComputation computation = statisticalManagerFactory
					.getComputation(computationId);
			SMOperationStatus status = SMOperationStatus.values()[computation
			                                                      .operationStatus()];
			switch (status) {
			case FAILED:
				throw new WorkerException("Failed Experiment.");
			case COMPLETED:
				complete = true;
				break;
			default:
				SMOperationInfo infos = statisticalManagerFactory
				.getComputationInfo(computationId, user);
				float smPercent = Float.parseFloat(infos.percentage()) / 100;
				updateProgress(0.10f + smPercent * 0.8f,"Waiting for computation to complete");
				try {
					Thread.sleep(5000);
				} catch (InterruptedException e) {
				}
			}
		}
	}

	private void generateTableFromResult() throws WorkerException {
		SMComputation computation = statisticalManagerFactory
				.getComputation(computationId);
		SMAbstractResource abstractResource = computation.abstractResource();
		SMResource smResource = abstractResource.resource();

		HashMap<String,String> metaMap=new HashMap<String,String>();

		handleSMResource(smResource);

		if(metaMap.size()>0)
			resultTable=cubeManager.modifyTableMeta(targetTable.getId()).setTableMetadata(new GenericMapMetadata(metaMap)).create();
		else resultTable=targetTable;
	}

	private Table importFromTableSpace(SMTable table)throws WorkerException{
		WorkerWrapper<DataWorker,WorkerResult> wrapper=new WorkerWrapper<DataWorker,WorkerResult>(importFactory);
		try{
			HashMap<String,Object> params= new HashMap<String, Object>();

			params.put(ImportFromStatisticalOperationFactory.RESOURCE_ID.getIdentifier(), table.resourceId());

			WorkerStatus status=wrapper.execute(null, null, getSourceInvocation().getParameterInstances());
			if(!status.equals(WorkerStatus.SUCCEDED)) throw new WorkerException("Failed export to dataspace");
			return wrapper.getResult().getResultTable();
		}catch(InvalidInvocationException e){
			throw new WorkerException("Unable to export table to dataspace.",e);
		} 
	}


	private void handleSMResource(SMResource toHandle) throws WorkerException{
		try{
			int resourceTypeIndex = toHandle.resourceType();
		SMResourceType smResType = SMResourceType.values()[resourceTypeIndex];
		switch (smResType) {
		case FILE:
			SMFile smFile=(SMFile)toHandle;
			results.add(new ImmutableURIResult(new InternalURI(new URI(smFile.url())), smFile.name(), smFile.description(), ResourceType.GENERIC_FILE));					
			break;
		case OBJECT:
			SMObject smObject=(SMObject) toHandle;
			if (smObject.name().contentEquals(PrimitiveTypes.MAP.toString())) {
				results.addAll(getFilesUrlFromFolderUrl(smObject.url()));
			}else 
				results.add(new ImmutableURIResult(new InternalURI(new URI(smObject.url())),smObject.name(),smObject.description(), ResourceType.GENERIC_FILE));
			break;
		case TABULAR:
			Table table=importFromTableSpace((SMTable) toHandle);
			results.add(new ImmutableTableResource(new TableResource(table.getId()),OperationHelper.retrieveTableLabel(table),"Imported from SM",ResourceType.GENERIC_TABLE));
			break;
		}
		}catch(Exception e){
			logger.warn("Unable to get resource "+toHandle,e);
		}
	}






	//	private Table addMapMetadata(Table toEnrich,Map<String,String> theMap){
	//		TableMetaCreator tmc = cubeManager.modifyTableMeta(toEnrich.getId());
	//		tmc.setTableMetadata(new GenericMapMetadata(theMap));
	//		return tmc.create();
	//	}




	private void importIntoDataSpace() throws WorkerException{
		WorkerWrapper<ResourceCreatorWorker,ResourcesResult> wrapper=new WorkerWrapper<ResourceCreatorWorker,ResourcesResult>(exportFactory);
		try{
			WorkerStatus status=wrapper.execute(targetTable.getId(), null, getSourceInvocation().getParameterInstances());
			if(!status.equals(WorkerStatus.SUCCEDED)) throw new WorkerException("Failed export to dataspace");
			String dataSpaceUri=wrapper.getResult().getResources().get(0).getResource().getStringValue();
			dataSpaceTableId=dataSpaceUri.substring(dataSpaceUri.indexOf(':')+1);
		}catch(InvalidInvocationException e){
			throw new WorkerException("Unable to export table to dataspace.",e);
		} catch (NoSuchMetadataException e) {
			throw new WorkerException("Unable to get dataspace table id for exported table",e);
		}
	}


	private List<ResourceDescriptorResult> getFilesUrlFromFolderUrl(String url) throws WorkerException{

		try{
			ArrayList<ResourceDescriptorResult> toReturn=new ArrayList<ResourceDescriptorResult>();

			Workspace ws = home.getWorkspace();
			WorkspaceItem folderItem = ws.getItemByPath(url);

			WorkspaceFolder folder = (WorkspaceFolder) folderItem;
			List<WorkspaceItem> childrenList = folder.getChildren();
			for (WorkspaceItem item : childrenList) {
				ExternalImage file = (ExternalImage) item;
				String name = item.getName();
				String absoluteUrlFile = file.getPublicLink();

				toReturn.add(new ImmutableURIResult(new InternalURI(new URI(absoluteUrlFile)), name, item.getDescription(), ResourceType.GENERIC_FILE));
			}

			return toReturn;
		}catch(Exception e){
			throw new WorkerException("Unable to retrieve results from workspace ",e);
		}
	}


}
