package org.gcube.data.analysis.tabulardata.operation.utils;

import java.util.ArrayList;
import java.util.List;

import org.gcube.common.database.DatabaseEndpointIdentifier;
import org.gcube.common.database.DatabaseProvider;
import org.gcube.common.database.endpoint.DatabaseEndpoint;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.model.metadata.common.TableDescriptorMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.ImmutableInlineTable;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.InlineColumn;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.InlineRow;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.InlineTable;
import org.gcube.data.analysis.tabulardata.operation.worker.results.resources.ResourceDescriptorResult;
import org.gcube.data.analysis.tabulardata.operation.worker.types.ResourceCreatorWorker;
import org.gcube.dataanalysis.lexicalmatcher.analysis.core.LexicalEngineConfiguration;
import org.gcube.dataanalysis.lexicalmatcher.analysis.guesser.data.Category;
import org.gcube.dataanalysis.lexicalmatcher.analysis.guesser.data.SingleResult;
import org.gcube.dataanalysis.lexicalmatcher.analysis.run.CategoryGuesser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GuesserWorker extends ResourceCreatorWorker{

	private static Logger logger = LoggerFactory.getLogger(GuesserWorker.class);
	
	DatabaseProvider database;
	CubeManager cubeManager;
		
	public GuesserWorker(OperationInvocation sourceInvocation, DatabaseProvider database, CubeManager cubeManager) {
		super(sourceInvocation);
		this.database = database;
		this.cubeManager = cubeManager;
	}

	@Override
	protected ResourceDescriptorResult execute() throws WorkerException {
		
		LexicalEngineConfiguration configurator = new LexicalEngineConfiguration();
		
		DatabaseEndpoint dbInstance = database.get(new DatabaseEndpointIdentifier("TabularData Database","Data-User"));
		
		configurator.setDatabaseUserName(dbInstance.getCredentials().getUsername());
		configurator.setDatabasePassword(dbInstance.getCredentials().getPassword());
		configurator.setDatabaseDriver("org.postgresql.Driver");
		configurator.setDatabaseURL(dbInstance.getConnectionString());
		configurator.setDatabaseDialect("org.hibernate.dialect.PostgreSQLDialect");
		
		ArrayList<Category> categories = retrieveCategories();
		configurator.setCategories(categories);
		CategoryGuesser guesser = new CategoryGuesser();
		Table targetTable= cubeManager.getTable(getSourceInvocation().getTargetTableId());
		try {
			guesser.runGuesser(targetTable.getName(), targetTable.getColumnById(getSourceInvocation().getTargetColumnId()).getName(), configurator);
		} catch (Exception e) {
			logger.error("error executing guesser",e);
			throw new WorkerException("error executing guesser",e);
		}
		ArrayList<SingleResult> results = guesser.getLastResults();
		
		logger.trace("guesser results are  "+results.size());
		
		logger.trace("guesser result: "+getOutput(results).getJSON());
						
		return new ImmutableInlineTable(getOutput(results),"guesser", "guesser");
	}

	private InlineTable getOutput(ArrayList<SingleResult> results) {
		InlineTable table = new InlineTable(new InlineColumn<Long>("tableId", Long.class), new InlineColumn<String>("columnId", String.class),  
				new InlineColumn<Double>("score", Double.class));
		for (SingleResult result : results){
			long codelistId = Long.parseLong(result.getFamilyID());
			Table codelist = cubeManager.getTable(new TableId(codelistId));
			String localId = codelist.getColumnByName(result.getColumn()).getLocalId().getValue();
			table.addRow(new InlineRow(codelistId, localId, result.getScore() ));
		}
		return table;
	}

	@SuppressWarnings("unchecked")
	private ArrayList<Category> retrieveCategories() {
		ArrayList<Category> categories = new ArrayList<Category>();
				
		List<Integer> codelistIds = (List<Integer>) getSourceInvocation().getParameterInstances().get(GuesserWorkerFactory.CODELISTS_PARAMETER.getIdentifier());
		for (Integer id :codelistIds ){
			Table table = cubeManager.getTable(new TableId(id));
			if (table.contains(TableDescriptorMetadata.class)){
				TableDescriptorMetadata tdm = table.getMetadata(TableDescriptorMetadata.class);
				categories.add(new Category(tdm.getName(), table.getId().getValue()+"", table.getName(), tdm.getName()));
			} else 
				categories.add(new Category("undefined", table.getId().getValue()+"", table.getName(), "undefined"));
		}
		return categories;
	}

	
	
}
