package org.gcube.data.analysis.tabulardata.operation.validation;

import java.util.ArrayList;
import java.util.List;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.column.factories.ValidationColumnFactory;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.ValidationColumnType;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataValidationMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ValidationReferencesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.table.GlobalDataValidationReportMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.ValidationHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DuplicateRowValidator extends Worker {
	
	private static final Logger log = LoggerFactory.getLogger(DuplicateRowValidator.class);

	CubeManager cubeManager;

	DatabaseConnectionProvider connectionProvider;

	SQLExpressionEvaluatorFactory evaluatorFactory;
	
	Table targetTable;

	Column validationColumn;

	List<Column> toCheckColumns=null;
	
	public DuplicateRowValidator(OperationInvocation sourceInvocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,SQLExpressionEvaluatorFactory evaluatorFactory) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.evaluatorFactory=evaluatorFactory;
	}

	@Override
	protected WorkerResult execute() throws WorkerException {
		retrieveTargetTable();
		updateProgress(0.2f);
		createNewTableWithValidationColumn();
		updateProgress(0.4f);
		fillNewTableWithData();
		updateProgress(0.8f);
		evaluateValidityAndUpdateTableMeta();
		return new ImmutableWorkerResult(targetTable);
	}

	private void evaluateValidityAndUpdateTableMeta() throws WorkerException {
		try{
			int invalidCount=ValidationHelper.getErrorCount(connectionProvider, targetTable, validationColumn, evaluatorFactory);
			GlobalDataValidationReportMetadata globalMeta=ValidationHelper.createDataValidationReport(validationColumn);
			
			targetTable = cubeManager.modifyTableMeta(targetTable.getId())
					.setColumnMetadata(validationColumn.getLocalId(), 
							createDataValidationMetadata(invalidCount)).
							setTableMetadata(globalMeta).create();
			}catch(Exception e){
				throw new WorkerException("Unable to evaluate global validation",e);
			}
	}

	private void fillNewTableWithData() throws WorkerException {
		try {
			SQLHelper.executeSQLBatchCommands(connectionProvider, createSetAllTrueSQL(), createSetDuplicateASFalse());
		} catch (Exception e) {
			String msg = "Unable to perform SQL operation";
			log.error(msg,e);
			throw new WorkerException(msg);
		}
	}

	private String createSetAllTrueSQL() {
		return String.format("UPDATE %1$s as newtable SET %2$s = true;",targetTable.getName(), validationColumn.getName());
	}

	private String createSetDuplicateASFalse() {
		
		String columnNamesSnippet = OperationHelper.getColumnNamesSnippet(toCheckColumns);
		return String.format("WITH duplicates AS (" +
				"Select %1$s from %2$s group by %1$s  having count(*)>1)," +
				"firsts AS (" +
				"Select distinct(first_value(%2$s .id) over (partition by %1$s )) AS id  from %2$s  where (%1$s)  IN (Select %1$s  from duplicates))" +
				"Update %2$s  SET %3$s = false WHERE id not in (SELECT id from firsts) AND (%1$s) in (SELECT %1$s  from duplicates)",columnNamesSnippet,targetTable.getName(),validationColumn.getName());
		
	}


	
	private void createNewTableWithValidationColumn() {
		DataValidationMetadata dataValidationMetadata = createDataValidationMetadata(0);
		validationColumn = new ValidationColumnFactory().create(new ImmutableLocalizedText("Unique Column Set"),dataValidationMetadata);
		targetTable=cubeManager.addValidations(targetTable.getId(),validationColumn);
		
		ValidationReferencesMetadata referencesMetadata=new ValidationReferencesMetadata(toCheckColumns.toArray(new Column[toCheckColumns.size()]));
		
		targetTable=cubeManager.modifyTableMeta(targetTable.getId()).setColumnMetadata(validationColumn.getLocalId(), referencesMetadata).create();
		
	}

	private DataValidationMetadata createDataValidationMetadata(int count) {
		return new DataValidationMetadata(new ImmutableLocalizedText(String.format("True when %s is unique",OperationHelper.getColumnLabelsSnippet(toCheckColumns))),count);
	}

	
	
	@SuppressWarnings("unchecked")
	private void retrieveTargetTable() {
		targetTable = cubeManager.getTable(getSourceInvocation().getTargetTableId());
		if(getSourceInvocation().getParameterInstances().containsKey(DuplicateRowValidatorFactory.KEY.getIdentifier())){
			this.toCheckColumns=new ArrayList<Column>();
			Object colParam=getSourceInvocation().getParameterInstances().get(DuplicateRowValidatorFactory.KEY.getIdentifier());
			if(colParam instanceof Iterable<?>){
				for(ColumnReference col:(Iterable<ColumnReference>)colParam)
					this.toCheckColumns.add(targetTable.getColumnById(col.getColumnId()));
			}else{
				this.toCheckColumns.add(targetTable.getColumnById(((ColumnReference)colParam).getColumnId()));
			}			
		}else this.toCheckColumns=targetTable.getColumnsExceptTypes(IdColumnType.class, ValidationColumnType.class);	
	}

}
