package org.gcube.data.analysis.tabulardata.operation.validation;

import java.sql.SQLException;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.factories.ValidationColumnFactory;
import org.gcube.data.analysis.tabulardata.model.datatype.DataType;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataValidationMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ValidationReferencesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.common.LocalizedText;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.ValidationHelper;
import org.gcube.data.analysis.tabulardata.operation.datatype.TypeTransitionSQLHandler;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ColumnTypeCastValidator extends Worker {

	private static final Logger log = LoggerFactory.getLogger(ColumnTypeCastValidator.class);

	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;

	private Table targetTable;

	private Column targetColumn;

	private DataType targetType;

	private Column validationColumn;

	public ColumnTypeCastValidator(OperationInvocation sourceInvocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
	}

	@Override
	protected WorkerResult execute() throws WorkerException {
		retrieveParameters();
		updateProgress(0.1f);
		addValidationColumnToTable();
		updateProgress(0.4f);
		fillNewTableWithData();
		updateProgress(0.8f);
		createUpdatedTableMeta();
		return new ImmutableWorkerResult(targetTable);
	}

	private void retrieveParameters() {
		targetTable = cubeManager.getTable(getSourceInvocation().getTargetTableId());
		targetColumn = targetTable.getColumnById(getSourceInvocation().getTargetColumnId());
		targetType = OperationHelper.getParameter(ColumnTypeCastValidatorFactory.TARGET_TYPE_PARAMETER,
				getSourceInvocation());
	}

	private void addValidationColumnToTable() {
		createValidationColumn();
		targetTable=cubeManager.addValidations(targetTable.getId(),validationColumn);
		//link validated column to validation column
		
		ValidationReferencesMetadata referencesMeta;
		if (targetColumn.contains(ValidationReferencesMetadata.class)){
			referencesMeta = targetColumn.getMetadata(ValidationReferencesMetadata.class);
			referencesMeta.add(validationColumn.getLocalId());
		} else referencesMeta = new ValidationReferencesMetadata(validationColumn.getLocalId());
		
		targetTable=cubeManager.modifyTableMeta(targetTable.getId()).setColumnMetadata(targetColumn.getLocalId(), referencesMeta).create();
		
	}

	private void createValidationColumn() {
		String columnLabel = retrieveTargetColumnLabel();
		DataValidationMetadata validationMeta = createDataValidationMetadata(false);
		validationColumn = ValidationColumnFactory.create(
				new ImmutableLocalizedText(String.format("Is %s a valid %s?", columnLabel, targetType.getName())),
				validationMeta);
	}

	private DataValidationMetadata createDataValidationMetadata(boolean valid) {
		String columnLabel = retrieveTargetColumnLabel();
		LocalizedText validationText = new ImmutableLocalizedText(String.format(
				"Tells whether %s can be casted to a %s value", columnLabel, targetType.getName()));
		DataValidationMetadata validationMeta = new DataValidationMetadata(validationText, valid);
		return validationMeta;
	}

	private String retrieveTargetColumnLabel() {
		return OperationHelper.retrieveColumnLabel(targetColumn);
	}

	private void fillNewTableWithData() throws WorkerException {
		String sqlCommand = generateFillTableSQL();
		try {
			SQLHelper.executeSQLCommand(sqlCommand, connectionProvider);
		} catch (SQLException e) {
			throw new WorkerException("Unable to evaluate validation result", e);
		}
	}

	private String generateFillTableSQL() {
		TypeTransitionSQLHandler typeTransitionHandler = TypeTransitionSQLHandler.getHandler(
				targetColumn.getDataType(), targetType);
		log.debug("Using Type transition handler: " + typeTransitionHandler.getClass().getSimpleName());
		return typeTransitionHandler.getFillValidationColumnSQLCommand(targetTable, validationColumn, targetColumn);
	}

	private void createUpdatedTableMeta() throws WorkerException {
		boolean valid = ValidationHelper.evaluateValidationColumnValidity(connectionProvider, targetTable.getName(),
				validationColumn.getName());
		targetTable = cubeManager.modifyTableMeta(targetTable.getId())
				.setColumnMetadata(validationColumn.getLocalId(), createDataValidationMetadata(valid)).create();
	}

}
