package org.gcube.data.analysis.tabulardata.operation.validation;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.expression.composite.ExternalReferenceExpression;
import org.gcube.data.analysis.tabulardata.expression.evaluator.description.DescriptionExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.expression.logical.ValueIsIn;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeNameColumnType;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDBoolean;
import org.gcube.data.analysis.tabulardata.model.table.TableType;
import org.gcube.data.analysis.tabulardata.model.table.type.CodelistTableType;
import org.gcube.data.analysis.tabulardata.operation.OperationId;
import org.gcube.data.analysis.tabulardata.operation.factories.types.ColumnValidatorFactory;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.ExpressionParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.TargetColumnParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;

@Singleton
public class ValidateDimensionColumnFactory extends ColumnValidatorFactory {

	private static final OperationId OPERATION_ID=new OperationId(5010);
	
	private CubeManager cubeManager;
	private DatabaseConnectionProvider connectionProvider;
	private SQLExpressionEvaluatorFactory sqlEvaluatorFactory;
	private DescriptionExpressionEvaluatorFactory descriptionEvaluatorFactory;
	
	public final static TargetColumnParameter TARGET_COLUMN_PARAMETER;
	
	static{
		List<Class<? extends TableType>> tableList = new ArrayList<Class<? extends TableType>>(1);
		tableList.add(CodelistTableType.class);
		TARGET_COLUMN_PARAMETER = new TargetColumnParameter("refColumn", "Codelist referenced column",
				"A codelist column containing values that are contained in the target column", Cardinality.ONE, 
				tableList,
				Arrays.asList(CodeColumnType.class,CodeNameColumnType.class));
	}
	
	public static final ExpressionParameter EXTERNAL_CONDITION_PARAMETER=new ExpressionParameter("expression", "Expression",
			"Condition on codelist values", Cardinality.OPTIONAL);
	
	private static List<Parameter> parameters=Arrays.asList(new Parameter[]{
		TARGET_COLUMN_PARAMETER,
		EXTERNAL_CONDITION_PARAMETER
	});
	
	@Override
	protected String getOperationName() {
		return "Dimension Column Validator";
	}

	@Override
	protected String getOperationDescription() {
		return "Validate the specified dimension column";
	}

	@Override
	protected List<Parameter> getParameters() {
		return parameters;
	}

	
	@Override
	protected OperationId getOperationId() {
		return OPERATION_ID;
	}

	@Inject
	public ValidateDimensionColumnFactory(CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,
			SQLExpressionEvaluatorFactory sqlEvaluatorFactory,
			DescriptionExpressionEvaluatorFactory descriptionEvaluatorFactory) {
		super();
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.sqlEvaluatorFactory = sqlEvaluatorFactory;
		this.descriptionEvaluatorFactory = descriptionEvaluatorFactory;
	}
	
	@Override
	public Worker createWorker(OperationInvocation invocation)
			throws InvalidInvocationException {
		performBaseChecks(invocation);
		Expression toCheck=generateValidationExpression(invocation);
		invocation.getParameterInstances().put(ValidateDataWithExpressionFactory.EXPRESSION_PARAMETER.getIdentifier(), toCheck);
		invocation.getParameterInstances().put(ValidateDataWithExpressionFactory.DESCRIPTION_PARAMETER.getIdentifier(), "External reference check");
		return new ValidateDataWithExpression(invocation, cubeManager, connectionProvider, sqlEvaluatorFactory,
				descriptionEvaluatorFactory);
	}
	
	public static Expression generateValidationExpression(OperationInvocation invocation)throws InvalidInvocationException{
		try{
			Map<String,Object> params=invocation.getParameterInstances();
			ColumnReference referredColumn = (ColumnReference) params.get(TARGET_COLUMN_PARAMETER.getIdentifier());
			Expression optionalCondition =null;
			if(params.containsKey(EXTERNAL_CONDITION_PARAMETER.getIdentifier()))optionalCondition=(Expression) params.get(EXTERNAL_CONDITION_PARAMETER.getIdentifier());
			else optionalCondition=new TDBoolean(true);
			ExternalReferenceExpression externalRef=new ExternalReferenceExpression(referredColumn, optionalCondition);
			ColumnReference targetReference=new ColumnReference(invocation.getTargetTableId(), invocation.getTargetColumnId());
			return new ValueIsIn(targetReference, externalRef);
		}catch(Exception e){
			throw new InvalidInvocationException(invocation, "Unable to generate validation expression", e);
		}
		
	}
	
}
