package org.gcube.data.analysis.tabulardata.operation.validation;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.factories.ValidationColumnFactory;
import org.gcube.data.analysis.tabulardata.model.metadata.column.DataValidationMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ValidationReferencesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.ValidationHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DuplicateValuesInColumnValidator extends Worker{

	private static final Logger log = LoggerFactory.getLogger(DuplicateValuesInColumnValidator.class);
	
	private CubeManager cubeManager;
	
	private DatabaseConnectionProvider connectionProvider;

	private Table targetTable;

	private Column targetColumn;

	private Column validationColumn;
	
	
	public DuplicateValuesInColumnValidator(
			OperationInvocation sourceInvocation, CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider) {
		super(sourceInvocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
	}

	@Override
	protected WorkerResult execute() throws WorkerException {
		retrieveParameters();
		updateProgress(0.3f);
		createNewTableWithValidationColumn();
		updateProgress(0.5f);
		fillNewTableWithData();
		updateProgress(0.8f);
		evaluateValidityAndUpdateTableMeta();
		return new ImmutableWorkerResult(targetTable);
	}
	
	private void retrieveParameters() {		
		targetTable = cubeManager.getTable(getSourceInvocation().getTargetTableId());
		targetColumn = targetTable.getColumnById(getSourceInvocation().getTargetColumnId());
	}
	
	
	private void createNewTableWithValidationColumn(){
		DataValidationMetadata dataValidationMetadata = createDataValidationMetadata(false);
		validationColumn = ValidationColumnFactory.create(new ImmutableLocalizedText("Not duplicate"),dataValidationMetadata);
		targetTable=cubeManager.addValidations(targetTable.getId(),validationColumn);
		
		ValidationReferencesMetadata referencesMeta;
		if (targetColumn.contains(ValidationReferencesMetadata.class)){
			referencesMeta = targetColumn.getMetadata(ValidationReferencesMetadata.class);
			referencesMeta.add(validationColumn.getLocalId());
		} else referencesMeta = new ValidationReferencesMetadata(validationColumn.getLocalId());
		
		targetTable=cubeManager.modifyTableMeta(targetTable.getId()).setColumnMetadata(targetColumn.getLocalId(), referencesMeta).create();
	}
	
	private DataValidationMetadata createDataValidationMetadata(boolean valid) {
		return new DataValidationMetadata(new ImmutableLocalizedText("True when the value in column "+targetColumn.getName()+" is not a duplicate, false otherwise"),valid);
	}
	
	private void fillNewTableWithData() throws WorkerException {
		try {
			SQLHelper.executeSQLBatchCommands(connectionProvider,createSetAllTrueSQL(),createSetFalseOnDuplicatesSQL());
		} catch (Exception e) {
			String msg = "Unable to perform SQL operation";
			log.error(msg,e);
			throw new WorkerException(msg);
		}
	}
	
	private String createSetAllTrueSQL() {
		return String.format("UPDATE %1$s as newtable SET %2$s = true;",targetTable.getName(), validationColumn.getName());
	}
	
	private String createSetFalseOnDuplicatesSQL(){
		return String.format("UPDATE %1$s as newtable SET %2$s = false WHERE %3$s IN (SELECT %3$s FROM %1$s GROUP BY %3$s HAVING count(*)>1)", targetTable.getName(), validationColumn.getName(), targetColumn.getName());
	}
	
	private void evaluateValidityAndUpdateTableMeta() throws WorkerException {
		boolean valid = ValidationHelper.evaluateValidationColumnValidity(connectionProvider, targetTable.getName(), validationColumn.getName());
		DataValidationMetadata validationMetadata = createDataValidationMetadata(valid);
		targetTable = cubeManager.modifyTableMeta(targetTable.getId()).setColumnMetadata(validationColumn.getLocalId(), validationMetadata).create();
	}
}
