package org.gcube.data.analysis.tabulardata.operation.table;

import java.util.Collection;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableCreator;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableMetaCreator;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeDescriptionColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeNameColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.DimensionColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.TimeDimensionColumnType;
import org.gcube.data.analysis.tabulardata.model.metadata.column.ViewColumnMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.table.DatasetViewTableMetadata;
import org.gcube.data.analysis.tabulardata.model.relationship.TableRelationship;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.type.DatasetViewTableType;
import org.gcube.data.analysis.tabulardata.operation.SQLHelper;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.ImmutableWorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.WorkerResult;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.WorkerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CreateDatasetView extends Worker {

	private final static Logger log = LoggerFactory.getLogger(CreateDatasetView.class);

	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;

	private Table targetDataset;

	public CreateDatasetView(CubeManager cubeManager, DatabaseConnectionProvider connectionProvider,
			OperationInvocation invocation) {
		super(invocation);
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		targetDataset = cubeManager.getTable(invocation.getTargetTableId());
	}

	@Override
	protected WorkerResult execute() throws WorkerException {
		Table viewTable = createDatasetViewTable();
		updateProgress(0.1f);
		String sql = createSQLStatement(viewTable);
		updateProgress(0.3f);
		executeSQLCommand(sql);
		log.trace("Created view table:\n" + viewTable);
		Table resultTable = createResultTable(viewTable);
		return new ImmutableWorkerResult(resultTable);
	}

	private void executeSQLCommand(String sql)  throws WorkerException {
		try {
			SQLHelper.executeSQLCommand(sql, connectionProvider);
		} catch (Exception e) {
			throw new WorkerException("Error occurred while executing SQL Command", e);
		}
	}

	private Table createResultTable(Table viewTable) {
		TableMetaCreator tmc = cubeManager.modifyTableMeta(targetDataset.getId());
		tmc.setTableMetadata(new DatasetViewTableMetadata(viewTable.getId()));
		return tmc.create();
	}

	@SuppressWarnings("unchecked")
	private String createSQLStatement(Table viewTable) {
		//Collection<Table> linkedTables = collectLinkedTables();

		StringBuilder sqlSelect = new StringBuilder("SELECT ");
		appendDatasetColumnNames(sqlSelect);
		//appendLinkedTableColumnNames(linkedTables, sqlSelect);
		sqlSelect.append("FROM " + targetDataset.getName() + " AS d ");

		int i = 0;
		for (TableRelationship rel : targetDataset.getRelationships()) {
			Table linkedTable = cubeManager.getTable(rel.getTargetTableId());
			String linkedTableAlias = "c" + i;

			String linkedTableName = linkedTable.getName();
			String foreignKeyColumnName = targetDataset.getColumnById(rel.getForeignKeyColumnId()).getName();
			String targetIdColumnName = linkedTable.getColumnsByType(IdColumnType.class).get(0).getName();
			sqlSelect.append(String.format("LEFT JOIN %1$s AS %2$s ON d.%3$s = %2$s.%4$s", linkedTableName,
					linkedTableAlias, foreignKeyColumnName, targetIdColumnName));
			i++;
		}
		// Create INSERT statement
		StringBuilder orderedColumnNames = new StringBuilder();
		for (Column column : viewTable.getColumns()) {
			if (column.getColumnType().equals(new IdColumnType())) continue;
			orderedColumnNames.append(column.getName() + " ,");
		}
		orderedColumnNames.deleteCharAt(orderedColumnNames.length() - 1); // Remove
		// last
		// comma
		String sqlInsert = String.format("INSERT INTO %s (%s) ", viewTable.getName(), orderedColumnNames);

		StringBuilder sql =new StringBuilder(sqlInsert).append(sqlSelect).append(";");
		log.trace("Generated \"view filling\" SQL statement:\n" + sql);
		return sql.toString();
	}

	@SuppressWarnings("unchecked")
	private void appendDatasetColumnNames(StringBuilder sqlSelect) {
		int relIndex = 0;
		for (Column column : targetDataset.getColumns()) {
			if (column.getColumnType().equals(new IdColumnType()))
				continue;
			sqlSelect.append("d.").append(column.getName()).append(" ,");
			if (column.getRelationship()!=null){
				Table linkedTable = cubeManager.getTable(column.getRelationship().getTargetTableId());
				for (Column relColumn : linkedTable.getColumnsByType(CodeColumnType.class, CodeNameColumnType.class,
						CodeDescriptionColumnType.class))
					sqlSelect.append("c").append(relIndex).append(".").append(relColumn.getName()).append(" ,");
				relIndex++;
			}
		}
		sqlSelect.deleteCharAt(sqlSelect.length() - 1); // Remove last comma
	}

	@SuppressWarnings("unchecked")
	private Table createDatasetViewTable() throws WorkerException {
		try {
			TableCreator tableCreator = cubeManager.createTable(new DatasetViewTableType());
			tableCreator.like(targetDataset, false);
			Collection<Column> dimensionColumns = targetDataset.getColumnsByType(new DimensionColumnType(),
					new TimeDimensionColumnType());
			for (Column dimensionColumn : dimensionColumns) {
				Column afterColumn = dimensionColumn;
				Table dimensionTable = cubeManager.getTable(dimensionColumn.getRelationship().getTargetTableId());
				for (Column column : dimensionTable.getColumnsByType(CodeColumnType.class, CodeNameColumnType.class,
						CodeDescriptionColumnType.class)) {
					column.setMetadata(new ViewColumnMetadata(dimensionTable.getId(), column.getLocalId(),
							dimensionColumn.getLocalId()));
					tableCreator.addColumnAfter(column, afterColumn);
					afterColumn = column;
				}
			}
			Table viewTable = tableCreator.create();
			log.debug("Created view: " + viewTable);
			return viewTable;
		} catch (Exception e) {
			throw new WorkerException("Unable to create dataset view", e);
		}

	}

}
