package org.gcube.data.analysis.tabulardata.operation.table;

import java.util.List;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.model.table.type.DatasetTableType;
import org.gcube.data.analysis.tabulardata.operation.factories.types.TableTransformationWorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.worker.EligibleOperation;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor.OperationId;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.OperationNotEligibleException;

import com.google.common.collect.Lists;

@Singleton
public class CreateDatasetViewFactory extends TableTransformationWorkerFactory {

	private static final OperationId OPERATION_ID = new OperationId(1003);

	private static final List<Parameter> parameters = Lists.newArrayList();

	private CubeManager cubeManager;

	private DatabaseConnectionProvider connectionProvider;

	@Inject
	public CreateDatasetViewFactory(CubeManager cubeManager, DatabaseConnectionProvider connectionProvider) {
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
	}

	public EligibleOperation getEligibleOperation(TableId tableId) throws OperationNotEligibleException {
		return getEligibleOperation(cubeManager.getTable(tableId));
	}

	public EligibleOperation getEligibleOperation(Table table) throws OperationNotEligibleException {
		try {
			checkEligibility(table);
		} catch (Exception e) {
			throw new OperationNotEligibleException(table.getId(),e.getMessage()); 
		}
		return new EligibleOperation(getOperationDescriptor(), parameters, table.getId());
	}

	private void checkEligibility(Table table) throws Exception {
		if (!table.getTableType().equals(new DatasetTableType()))
			throw new Exception("Table is not a dataset");
		if (!table.hasRelationships())
			throw new Exception("Table has no relationship, a view is not needed");
	}

	public Worker createWorker(OperationInvocation invocation) throws InvalidInvocationException {
		checkInvocation(invocation);
		return new CreateDatasetView(cubeManager, connectionProvider, invocation);
	}

	private void checkInvocation(OperationInvocation invocation) throws InvalidInvocationException {
		Table table = cubeManager.getTable(invocation.getTargetTableId());
		
		try {
			checkEligibility(table);
		} catch (Exception e) {
			throw new InvalidInvocationException(invocation, "Target table is not eligible");
		}
		
	}

	@Override
	protected String getOperationName() {
		return "Create dataset view";
	}

	@Override
	protected String getOperationDescription() {
		return "Create a dataset materialized view with data from linked codelists";
	}

	@Override
	protected OperationId getOperationId() {
		return OPERATION_ID;
	}

}
