package org.gcube.data.analysis.tabulardata.operation.column;

import java.util.List;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.model.column.ColumnLocalId;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.TableId;
import org.gcube.data.analysis.tabulardata.operation.factories.types.ColumnTransformationWorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.worker.EligibleOperation;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationDescriptor.OperationId;
import org.gcube.data.analysis.tabulardata.operation.worker.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.OperationNotEligibleException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

@Singleton
public class RemoveColumnFactory extends ColumnTransformationWorkerFactory {

	private final static Logger log = LoggerFactory.getLogger(RemoveColumnFactory.class);

	private static final OperationId OPERATION_ID = new OperationId(1004);

	@Inject
	private CubeManager cubeManager;

	@Override
	public EligibleOperation getEligibleOperation(TableId tableId, ColumnLocalId columnId) throws OperationNotEligibleException {
		return getEligibleOperation(cubeManager.getTable(tableId), columnId);
	}

	
	private EligibleOperation getEligibleOperation(Table table, ColumnLocalId columnId) throws OperationNotEligibleException {
		try {
			checkTable(table, columnId);
			List<Parameter> parameters = Lists.newArrayList();
			return new EligibleOperation(getOperationDescriptor(), parameters, table.getId(), columnId);
		} catch (Exception e) {
			log.warn("Operation not eligible", e);
			throw new OperationNotEligibleException("Operation is not eligible");
		}
	}

	private void checkTable(Table table, ColumnLocalId columnId) {
		table.getColumnById(columnId);
	}

	@Override
	public Worker createWorker(OperationInvocation invocation) throws InvalidInvocationException {
		checkInvocation(invocation);
		return new RemoveColumn(cubeManager, invocation);
	}

	private void checkInvocation(OperationInvocation invocation) throws InvalidInvocationException {
		try {
			checkTable(cubeManager.getTable(invocation.getTargetTableId()), invocation.getTargetColumnId());
		} catch (Exception e) {
			throw new InvalidInvocationException(invocation, e);
		}
	}

	@Override
	protected String getOperationName() {
		return "Remove column";
	}

	@Override
	protected String getOperationDescription() {
		return "Remove column from table";
	}

	@Override
	protected OperationId getOperationId() {
		return OPERATION_ID;
	}

}
