package org.gcube.data.analysis.tabulardata.operation.data.add;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.ValidationColumnType;
import org.gcube.data.analysis.tabulardata.model.datatype.DataType;
import org.gcube.data.analysis.tabulardata.model.table.TableType;
import org.gcube.data.analysis.tabulardata.operation.OperationId;
import org.gcube.data.analysis.tabulardata.operation.factories.types.TableTransformationWorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.ColumnTypeParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.DataTypeParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.LocalizedTextParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.Worker;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;

@Singleton
public class AddColumnFactory extends TableTransformationWorkerFactory{

	private static final OperationId OPERATION_ID = new OperationId(1005);
	
	
	public static final DataTypeParameter DATA_TYPE=new DataTypeParameter("dataType", "Data Type", "To set data type", Cardinality.OPTIONAL);
	public static final ColumnTypeParameter COLUMN_TYPE=new ColumnTypeParameter("columnType","Column type","The type of the new column",Cardinality.ONE);
	public static final LocalizedTextParameter LABEL= new LocalizedTextParameter("label", "label", "To set label", Cardinality.OPTIONAL);
	
	
	
	private static List<Parameter> parameters=Arrays.asList(new Parameter[]{
		COLUMN_TYPE,
		DATA_TYPE,
		LABEL
	});
	
	
	@Inject
	private CubeManager cubeManager;
	@Inject
	private SQLExpressionEvaluatorFactory sqlEvaluatorFactory;
	@Inject
	private DatabaseConnectionProvider connectionProvider;
	
	
	@Override
	public Worker createWorker(OperationInvocation invocation) throws InvalidInvocationException {
		performBaseChecks(invocation,cubeManager);
		checkAllowedColumnDataCombination(invocation);
		return new AddColumn(invocation, cubeManager,sqlEvaluatorFactory,connectionProvider);
	}

	@Override
	protected String getOperationName() {
		return "Adds a column";
	}

	@Override
	protected String getOperationDescription() {
		return "Adds a column to a table";
	}

	@Override
	protected OperationId getOperationId() {
		return OPERATION_ID;
	}
	
	
	@Override
	protected List<Parameter> getParameters() {
		return parameters;
	}
	
	private void checkAllowedColumnDataCombination(OperationInvocation invocation) throws InvalidInvocationException{
		TableType tableType=cubeManager.getTable(invocation.getTargetTableId()).getTableType();
		Map<String,Object> params=invocation.getParameterInstances();
		ColumnType colType=(ColumnType) params.get(COLUMN_TYPE.getIdentifier());	
		if (colType.equals(new ValidationColumnType())||colType.equals(new IdColumnType()))
			throw new InvalidInvocationException(invocation,String.format("Column type %s cannot be added by user",colType.getName()));
		if(!tableType.getAllowedColumnTypes().contains(colType))	
			throw new InvalidInvocationException(invocation, String.format("Passed column type %s is not allowed for target table type %s. Allowed types are %s.", colType.getName(),tableType.getName(),tableType.getAllowedColumnTypes()));
		if(params.containsKey(DATA_TYPE.getIdentifier())){
			DataType type=(DataType) params.get(DATA_TYPE.getIdentifier());
			if(!colType.isDataTypeAllowed(type))
				throw new InvalidInvocationException(invocation, String.format("Incompatible column type %s and data type %s. Allowed data types are %s.", colType.getName(),type.getName(),colType.getAllowedDataTypes()));
		}
	}
}
