package org.gcube.data.analysis.tabulardata.operation.comet;

import java.util.Arrays;
import java.util.List;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.gcube.data.analysis.tabulardata.cube.CubeManager;
import org.gcube.data.analysis.tabulardata.cube.data.connection.DatabaseConnectionProvider;
import org.gcube.data.analysis.tabulardata.expression.evaluator.sql.SQLExpressionEvaluatorFactory;
import org.gcube.data.analysis.tabulardata.operation.OperationHelper;
import org.gcube.data.analysis.tabulardata.operation.OperationId;
import org.gcube.data.analysis.tabulardata.operation.OperationType;
import org.gcube.data.analysis.tabulardata.operation.factories.types.TableMetadataWorkerFactory;
import org.gcube.data.analysis.tabulardata.operation.invocation.OperationInvocation;
import org.gcube.data.analysis.tabulardata.operation.parameters.Cardinality;
import org.gcube.data.analysis.tabulardata.operation.parameters.Parameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.RegexpStringParameter;
import org.gcube.data.analysis.tabulardata.operation.parameters.leaves.TargetColumnParameter;
import org.gcube.data.analysis.tabulardata.operation.worker.exceptions.InvalidInvocationException;
import org.gcube.data.analysis.tabulardata.operation.worker.types.MetadataWorker;

@Singleton
public class ImportCodeListMappingFactory extends  TableMetadataWorkerFactory{
	
	private static final OperationId OPERATION_ID=new OperationId(12001);
	
	public static RegexpStringParameter URL_PARAMETER =  new RegexpStringParameter("url", "Document URL",
			"URL that points to a location where the document can be downloaded.", Cardinality.ONE,
			".*");
	
	public static TargetColumnParameter PREVIOUS_VERSION_CODELIST_PARAMETER=
			new TargetColumnParameter("old_codes", "Code column before curation", "The code column of the previous version of this codelist, from which the mapping is generated", 
					Cardinality.ONE);
	
	
	private static final List<Parameter> parameters=Arrays.asList(new Parameter[]{
			URL_PARAMETER,
			PREVIOUS_VERSION_CODELIST_PARAMETER
	});
	
	private CubeManager cubeManager;
	private DatabaseConnectionProvider connectionProvider;
	private SQLExpressionEvaluatorFactory evaluatorFactory;
	
		
	@Inject
	public ImportCodeListMappingFactory(CubeManager cubeManager,
			DatabaseConnectionProvider connectionProvider,
			SQLExpressionEvaluatorFactory evaluatorFactory) {
		super();
		this.cubeManager = cubeManager;
		this.connectionProvider = connectionProvider;
		this.evaluatorFactory = evaluatorFactory;
	}





	@Override
	public MetadataWorker createWorker(OperationInvocation arg0)
			throws InvalidInvocationException {
		performBaseChecks(arg0, cubeManager);
		checkUrl(arg0);
		return new ImportCodelistMappingWorker(arg0, cubeManager, connectionProvider, evaluatorFactory);
	}
	
	private void checkUrl(OperationInvocation invocation)throws InvalidInvocationException{
		try {
			OperationHelper.getInputStreamFromUrl(OperationHelper.getParameter(URL_PARAMETER, invocation));
		} catch (Exception e) {
			throw new InvalidInvocationException(invocation, "Unable to connect to mapping file.",e);
		}
	}
	
	
	
	@Override
	protected String getOperationDescription() {
		return "Imports a comet file and infers harmonization rules.";
	}

	@Override
	protected String getOperationName() {
		return "Import comet mapping";
	}
	
	@Override
	protected List<Parameter> getParameters() {
		return parameters;
	}


	@Override
	protected OperationType getOperationType() {
		return OperationType.RESOURCECREATOR;
	}
	
	@Override
	public OperationId getOperationId() {
		return OPERATION_ID;
	}
	
	
}