package org.gcube.data.analysis.tabulardata.cube.time;

import java.util.Arrays;
import java.util.List;

import javax.inject.Inject;

import org.gcube.data.analysis.tabulardata.cube.data.DatabaseWrangler;
import org.gcube.data.analysis.tabulardata.cube.metadata.CubeMetadataWrangler;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.factories.AttributeColumnFactory;
import org.gcube.data.analysis.tabulardata.model.column.factories.CodeColumnFactory;
import org.gcube.data.analysis.tabulardata.model.column.factories.IdColumnFactory;
import org.gcube.data.analysis.tabulardata.model.column.type.CodeColumnType;
import org.gcube.data.analysis.tabulardata.model.column.type.IdColumnType;
import org.gcube.data.analysis.tabulardata.model.datatype.IntegerType;
import org.gcube.data.analysis.tabulardata.model.datatype.TextType;
import org.gcube.data.analysis.tabulardata.model.metadata.common.ImmutableLocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.common.LocalizedText;
import org.gcube.data.analysis.tabulardata.model.metadata.common.NamesMetadata;
import org.gcube.data.analysis.tabulardata.model.metadata.table.TimePeriodTypeMetadata;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.type.TimeCodelistTableType;
import org.gcube.data.analysis.tabulardata.model.time.PeriodType;

import com.google.common.collect.Lists;

public class YearCodelistCreator implements TimeCodelistCreator {
	@Inject
	DatabaseWrangler dbWrangler;

	@Inject
	CubeMetadataWrangler cmWrangler;

	@Override
	public Table createTable() {
		Table codelist = createTableMeta();

		String tableName = createTableOnDBWithColumns(codelist.getColumns());
		codelist.setName(tableName);

		return cmWrangler.save(codelist, false);
	}

	private Table createTableMeta() {
		List<Column> columns = Lists.newArrayList(IdColumnFactory.create(), createCodeColumn(), createYearColumn(), createDecadeColumn());
		Table dayCodelist = new Table(new TimeCodelistTableType());
		dayCodelist.setColumns(columns);
		dayCodelist.setMetadata(new TimePeriodTypeMetadata(getManagedPeriodType()));
		return dayCodelist;
	}
	private Column createCodeColumn() {
		Column codeColumn = new CodeColumnFactory().createDefault();
		codeColumn.setName("year_code");
		codeColumn.setDataType(new TextType(4));
		return codeColumn;
	}

	private String createTableOnDBWithColumns(List<Column> columns) {
		String tableName = dbWrangler.createTable();
		for (Column column : columns) {
			if (column.getColumnType() instanceof IdColumnType)
				continue;
			dbWrangler.addColumn(tableName, column.getName(), column.getDataType());
			dbWrangler.setNullable(tableName, column.getName(), false);
			if (column.getColumnType() instanceof CodeColumnType)
				dbWrangler.createUniqueIndex(tableName, column.getName());
		}
		dbWrangler.createInsertUpdateTriggerOnTable("update_time_dimension_" + tableName, tableName,
				"update_gregorian_year_dimension");
		return tableName;
	}

	private Column createDecadeColumn() {
		Column resultColumn = new AttributeColumnFactory().create(new IntegerType());
		resultColumn.setName("decade");
		resultColumn.setMetadata(new NamesMetadata(Arrays.asList((LocalizedText)new ImmutableLocalizedText("decade"))));
		return resultColumn;
	}

	private Column createYearColumn() {
		Column yearColumn = new AttributeColumnFactory().create(new IntegerType());
		yearColumn.setName("year");
		yearColumn.setMetadata(new NamesMetadata(Arrays.asList((LocalizedText)new ImmutableLocalizedText("year"))));
		return yearColumn;
	}

	@Override
	public PeriodType getManagedPeriodType() {
		return PeriodType.YEAR;
	}
}
