package org.gcube.data.analysis.tabulardata.cube.tablemanagers.dataset;

import java.util.Collection;
import java.util.List;

import org.gcube.data.analysis.tabulardata.cube.data.DatabaseWrangler;
import org.gcube.data.analysis.tabulardata.cube.exceptions.TableCreationException;
import org.gcube.data.analysis.tabulardata.cube.metadata.CubeMetadataWrangler;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.DefaultTableCreator;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableManager;
import org.gcube.data.analysis.tabulardata.model.column.AttributeColumn;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnType;
import org.gcube.data.analysis.tabulardata.model.column.IdColumn;
import org.gcube.data.analysis.tabulardata.model.column.MeasureColumn;
import org.gcube.data.analysis.tabulardata.model.idioms.ColumnIsOfType;
import org.gcube.data.analysis.tabulardata.model.idioms.TransformColumn;
import org.gcube.data.analysis.tabulardata.model.table.Dataset;
import org.gcube.data.analysis.tabulardata.model.table.Table;

import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;

public class DatasetCreator extends DefaultTableCreator {

	// Columns
	private List<Column> newColumns = Lists.newArrayList();

	// These are used for cloning
	private Table clonedTable;
	private List<Column> columnsToRemove = Lists.newArrayList();
	private boolean withData;

	public DatasetCreator(DatabaseWrangler dbWrangler, CubeMetadataWrangler cmWrangler, TableManager tableManager) {
		super(dbWrangler, cmWrangler, tableManager);
	}

	@Override
	protected Collection<Column> getAllColumns() {
		List<Column> columns = Lists.newArrayList();
		if (clonedTable != null) {
			columns.addAll(clonedTable.getColumns());
			columns.removeAll(columnsToRemove);
		}
		while (columns.remove(new IdColumn()))
			;
		columns.addAll(newColumns);
		return columns;
	}

	@Override
	protected Collection<Column> getNewColumns() {
		return newColumns;
	}

	private Collection<AttributeColumn> getAttributeColumns() {
		Collection<Column> tmp = Collections2.filter(getAllColumns(), new ColumnIsOfType(ColumnType.ATTRIBUTE));
		return Collections2.transform(tmp, new TransformColumn<AttributeColumn>());
	}

	private Collection<MeasureColumn> getMeasureColumns() {
		Collection<Column> tmp = Collections2.filter(getAllColumns(), new ColumnIsOfType(ColumnType.MEASURE));
		return Collections2.transform(tmp, new TransformColumn<MeasureColumn>());
	}

	@Override
	protected Table createBaseTable(String tableName, Collection<Column> columns) {
		return new Dataset(tableName, columns);
	}

	@Override
	protected void checkConsistency() throws TableCreationException {
		try {
			// Verify label uniqueness
			checkDuplicateLabels(getAllColumns());
		} catch (Exception e) {
			throw new TableCreationException(e.getMessage());
		}

		// Verify minimum constraints
		if (getAttributeColumns().isEmpty() && getMeasureColumns().isEmpty())
			throw new TableCreationException(
					"Must provide at least one attribute column or measure column in order to create a dataset.");

		try {
			// Verify CodelistRefColumn IDs and referenced column names
			checkColumnsRelationship(getAllColumns());
		} catch (Exception e) {
			throw new TableCreationException(e.getMessage());
		}
	}

	@Override
	protected boolean isAllowedColumn(Column column) {
		switch (column.getColumnType()) {
		case ATTRIBUTE:
		case MEASURE:
		case SYSTEM:
		case CODELISTREF:
			return true;
		default:
			return false;
		}
	}

	@Override
	protected void addNewColumn(Column column) {
		newColumns.add(column);
	}

	@Override
	protected boolean isAllowedCloneableTable(Table table) {
		switch (table.getTableType()) {
		case DATASET:
			return true;
		default:
			return false;
		}
	}

	@Override
	protected Table getTableToClone() {
		return clonedTable;
	}

	@Override
	protected void setTableToClone(Table table) {
		clonedTable = table;
	}

	@Override
	protected boolean isCloneWithData() {
		return withData;
	}

	@Override
	protected void setCloneWithData(boolean cloneWithData) {
		withData = cloneWithData;
	}

	@Override
	protected Collection<Column> getColumnsToRemove() {
		return columnsToRemove;
	}

	@Override
	protected void setColumnsToRemove(Collection<Column> columns) {
		columnsToRemove = Lists.newArrayList(columns);
	}

	@Override
	protected void addIndexes(String tableName, Collection<Column> columns) {
		//Do nothing
	}

}
