package org.gcube.data.analysis.tabulardata.cube.tablemanagers.view;

import java.util.Collection;
import java.util.List;

import org.gcube.data.analysis.tabulardata.cube.data.DatabaseWrangler;
import org.gcube.data.analysis.tabulardata.cube.exceptions.TableCreationException;
import org.gcube.data.analysis.tabulardata.cube.metadata.CubeMetadataWrangler;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.DefaultTableCreator;
import org.gcube.data.analysis.tabulardata.cube.tablemanagers.TableManager;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.IdColumn;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.model.table.ViewTable;

import com.google.common.collect.Lists;

public class ViewTableCreator extends DefaultTableCreator {

	// Columns
	private List<Column> newColumns = Lists.newArrayList();

	// These are used for cloning
	private Table tableToClone;
	/**
	 * These columns are part of the cloned table and must not be present in the
	 * new table
	 */
	private List<Column> columnsToRemove = Lists.newArrayList();
	private boolean withData;

	public ViewTableCreator(DatabaseWrangler dw, CubeMetadataWrangler cmw, TableManager tm) {
		super(dw, cmw, tm);
	}

	@Override
	protected Collection<Column> getAllColumns() {
		List<Column> columns = Lists.newArrayList();
		if (tableToClone != null) {
			columns.addAll(tableToClone.getColumns());
			columns.removeAll(columnsToRemove);
		}
		while (columns.remove(new IdColumn()))
			;
		columns.addAll(newColumns);
		return columns;
	}

	@Override
	protected Collection<Column> getNewColumns() {
		return newColumns;
	}

	@Override
	protected Table createBaseTable(String tableName, Collection<Column> columns) {
		return new ViewTable(tableName, getAllColumns());
	}

	@Override
	protected void checkConsistency() throws TableCreationException {
		try {
			// Verify label uniqueness
			checkDuplicateLabels(getAllColumns());
		} catch (Exception e) {
			throw new TableCreationException(e.getMessage());
		}

		// Verify CodelistRefColumn IDs and referenced column names
		try {
			checkColumnsRelationship(getAllColumns());
		} catch (Exception e) {
			throw new TableCreationException(e.getMessage());
		}
	}

	@Override
	protected boolean isAllowedColumn(Column column) {
		switch (column.getColumnType()) {
		case ATTRIBUTE:
		case MEASURE:
		case SYSTEM:
		case CODELISTREF:
			return true;
		default:
			return false;
		}
	}

	@Override
	protected void addNewColumn(Column column) {
		newColumns.add(column);
	}

	@Override
	protected boolean isAllowedCloneableTable(Table table) {
		switch (table.getTableType()) {
		case DATASET:
		case VIEWTABLE:
			return true;
		default:
			return false;
		}
	}

	@Override
	protected Table getTableToClone() {
		return tableToClone;
	}

	@Override
	protected void setTableToClone(Table table) {
		tableToClone = table;
	}

	@Override
	protected boolean isCloneWithData() {
		return withData;
	}

	@Override
	protected void setCloneWithData(boolean cloneWithData) {
		withData = cloneWithData;
	}

	@Override
	protected Collection<Column> getColumnsToRemove() {
		return columnsToRemove;
	}

	@Override
	protected void setColumnsToRemove(Collection<Column> columns) {
		columnsToRemove = Lists.newArrayList(columns);
	}

	@Override
	protected void addIndexes(String tableName, Collection<Column> columns) {
		for (Column column : columns) {
			switch (column.getColumnType()) {
			case CODELISTREF:
			case ID:
				dbWrangler.createIndex(tableName, column.getName());
			default:
			}
		}
	}

	@Override
	public Table create() throws TableCreationException {

		checkConsistency();

		setColumnNames(getAllColumns());

		String tableName = null;
		if (getTableToClone() != null) {
			// Handle cloning
			tableName = dbWrangler.cloneTable(getTableToClone().getName(), isCloneWithData(), true);

			for (Column column : getColumnsToRemove()) {
				dbWrangler.removeColumn(tableName, column.getName());
			}

			for (Column column : getNewColumns()) {
				dbWrangler.addColumn(tableName, column.getName(), column.getDataType());
			}

		} else {
			// Handle simple creation of empty table
			tableName = dbWrangler.createTable(true);
			for (Column column : getNewColumns()) {
				dbWrangler.addColumn(tableName, column.getName(), column.getDataType());
			}
		}

		addIndexes(tableName, getAllColumns());

		Collection<Column> columns = getAllColumns();
		columns.add(new IdColumn());
		Table newTable = createBaseTable(tableName, columns);

		if (getTableToClone() != null) {
			// Set parent table id
			newTable.setParentTableId(getTableToClone().getId());

			// Clone metadata
			cloneMetadata(getTableToClone(), newTable);
		}

		// Register codelist on metadata DB
		return mdWrangler.save(newTable);

	}

}
