package org.gcube.data.analysis.statisticalmanager.experimentspace.computation;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.state.GCUBEWSResource;
import org.gcube.common.encryption.StringEncrypter;
import org.gcube.common.resources.gcore.ServiceEndpoint.AccessPoint;
import org.gcube.common.resources.gcore.ServiceEndpoint.Property;
import org.gcube.common.resources.gcore.utils.Group;
import org.gcube.data.analysis.statisticalmanager.SMOperationStatus;
import org.gcube.data.analysis.statisticalmanager.ServiceContext;

import org.gcube.data.analysis.statisticalmanager.experimentspace.AlgorithmCategory;
import org.gcube.data.analysis.statisticalmanager.experimentspace.ComputationFactory;
import org.gcube.data.analysis.statisticalmanager.persistence.DataBaseManager;
import org.gcube.data.analysis.statisticalmanager.persistence.RemoteStorage;
import org.gcube.data.analysis.statisticalmanager.persistence.RuntimeResourceManager;
import org.gcube.data.analysis.statisticalmanager.persistence.SMPersistenceManager;
import org.gcube.data.analysis.statisticalmanager.stubs.SMAlgorithm;
import org.gcube.data.analysis.statisticalmanager.stubs.SMComputationConfig;
import org.gcube.data.analysis.statisticalmanager.util.ServiceUtil;
import org.gcube.dataanalysis.ecoengine.configuration.AlgorithmConfiguration;
import org.gcube.dataanalysis.ecoengine.configuration.INFRASTRUCTURE;
import org.gcube.dataanalysis.ecoengine.datatypes.DatabaseType;
import org.gcube.dataanalysis.ecoengine.datatypes.PrimitiveType;
import org.gcube.dataanalysis.ecoengine.datatypes.ServiceType;
import org.gcube.dataanalysis.ecoengine.datatypes.StatisticalType;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.DatabaseParameters;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.PrimitiveTypes;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.ServiceParameters;
import org.gcube.dataanalysis.ecoengine.interfaces.ComputationalAgent;
import org.gcube.dataanalysis.ecoengine.processing.factories.ClusterersFactory;
import org.gcube.dataanalysis.ecoengine.processing.factories.EvaluatorsFactory;
import org.gcube.dataanalysis.ecoengine.processing.factories.GeneratorsFactory;
import org.gcube.dataanalysis.ecoengine.processing.factories.ModelersFactory;
import org.gcube.dataanalysis.ecoengine.processing.factories.TransducerersFactory;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.ComputationalAgentClass;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMComputation;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMEntry;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMFile;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMInputEntry;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMResource;
import org.globus.wsrf.ResourceProperty;

public class ComputationResource extends GCUBEWSResource {

	private static final String NAME_RP_NAME = "computation";
	private volatile String userLogin;
	private volatile String scope;

	private static ConcurrentHashMap<String, ComputationalAgent> runningCAgents = new ConcurrentHashMap<String, ComputationalAgent>();

	private String getConfigPath() {
		return ServiceContext.getContext().getProperty("configDir") + "/cfg/";
	}

	public void cleanResourcesComputational(final INFRASTRUCTURE compInf,
			final long computationId) {

		switch (compInf) {
		case LOCAL:
			ComputationFactory.getFactoryResource().cleanLocalComputation();
			logger.debug("---------- Clean up local resources");
			// ComputationFactory.getFactoryResource().
			// cleanLocalResourcesComputational(String.valueOf(computationId));
			break;
		case D4SCIENCE:
			logger.debug("--------- Clen up D4Science resources");
			ComputationFactory.getFactoryResource().cleanD4ScienceComputation();
		default:
			break;
		}

	}

	private synchronized void addComputationalAgent(String key,
			ComputationalAgent agent) {

		runningCAgents.put(key, agent);
		ResourceProperty property = getResourcePropertySet().get(NAME_RP_NAME);
		property.add(key);
	}

	private synchronized void removeComputationalAgent(String key) {

		runningCAgents.remove(key);
		ResourceProperty property = getResourcePropertySet().get(NAME_RP_NAME);
		property.clear();
		for (Entry<String, ComputationalAgent> entry : runningCAgents
				.entrySet()) {
			property.add(entry.getKey());
		}

	}

	private void setComputationOutput(long computationId,
			ComputationalAgent agent) throws Exception {

		SMComputation computation = (SMComputation) SMPersistenceManager
				.getOperation(computationId);
		StatisticalType output = agent.getOutput();
		BuilderComputationOutput builder = new BuilderComputationOutput(
				userLogin, scope, computation);
		SMResource resource = builder.serialize(output);
		logger.debug("created SMResource output");
		logger.debug("save resource in db.......");
		SMPersistenceManager.addCreatedResource(computationId, resource);
	}

	/** {@inheritDoc} */
	@Override
	public void initialise(Object... args) throws Exception {

		logger.debug("------ initialize Service Resource");
		userLogin = (String) args[0];
		scope = (String) args[1];

	}

	/** {@inheritDoc} */
	@Override
	protected String[] getPropertyNames() {
		return new String[] { NAME_RP_NAME };
	}

	public ConcurrentHashMap<String, ComputationalAgent> getComputationalAgents() {
		return runningCAgents;
	}

	private AlgorithmConfiguration setUserParameters(
			AlgorithmConfiguration algoConfig, SMComputationConfig requestConfig)
			throws Exception {

		Map<String, StatisticalType> parameters = getMapParameters(requestConfig
				.getAlgorithm());

		// Set user algorithms parameters
		for (SMInputEntry parameter : requestConfig.getParameters().getList()) {
			logger.debug("Set Parameter user key " + parameter.getKey()
					+ " value " + parameter.getValue());
			String value = parameter.getValue();

			StatisticalType typeParam = parameters.get(parameter.getKey());
			if ((typeParam instanceof PrimitiveType)
					&& (((PrimitiveType) typeParam).getType() == PrimitiveTypes.FILE)) {

				SMFile file = SMPersistenceManager
						.getFile(parameter.getValue());
				logger.debug("Parameter is a file with id "
						+ parameter.getValue());
				logger.debug("File with name" + file.getName());

				RemoteStorage storage = new RemoteStorage(userLogin, scope);
				File tmpFile = storage.getFile(File.separator
						+ file.getRemoteName());
				logger.debug("----------- File created "
						+ tmpFile.getAbsolutePath());

				value = tmpFile.getAbsolutePath();
			}

			algoConfig.setParam(parameter.getKey(), value);
		}
		return algoConfig;
	}

	private Map<String, StatisticalType> getMapParameters(String algorithm)
			throws Exception {
		Map<String, StatisticalType> map = new HashMap<String, StatisticalType>();
		List<StatisticalType> list = getListParameters(algorithm);
		for (StatisticalType parameter : list) {
			map.put(parameter.getName(), parameter);
		}
		return map;
	}

	public List<StatisticalType> getListParameters(String algorithm)
			throws Exception {

		switch (AlgorithmCategory.valueOf(ServiceUtil
				.getAlgorithmCategory(algorithm))) {
		case DISTRIBUTIONS:
			return GeneratorsFactory.getAlgorithmParameters(getConfigPath(),
					algorithm);

		case EVALUATORS:
			return EvaluatorsFactory.getEvaluatorParameters(getConfigPath(),
					algorithm);

		case MODELS:
			return ModelersFactory.getModelParameters(getConfigPath(),
					algorithm);

		case TRANSDUCERS:
			return TransducerersFactory.getTransducerParameters(
					getConfigPath(), algorithm);

		case CLUSTERERS:
			return ClusterersFactory.getClustererParameters(getConfigPath(),
					algorithm);
		default:
			throw new Exception();
		}

	}

	private void setServiceParameters(SMComputationConfig computationConfig,
			AlgorithmConfiguration algoConfig, List<StatisticalType> parameters)
			throws SMParametersSettingException {

		logger.debug("Parameter retrieved " + parameters.size());

		String rrEndPoint = null;
		String rrUser = null;
		String rrPassword = null;
		String rrDBName = null;
		String rrDBDriver = null;
		String rrDBDialect = null;

		for (StatisticalType parameter : parameters) {
			if (parameter.getClass() == DatabaseType.class) {
				DatabaseType dbType = (DatabaseType) parameter;
				if (dbType.getDatabaseParameter() == DatabaseParameters.REMOTEDATABASERRNAME) {
					try {
						Group<AccessPoint> accessPoints = RuntimeResourceManager
								.getRRAccessPoint(dbType.getName());
						for (AccessPoint accessPoint : accessPoints) {
							rrEndPoint = accessPoint.address();
							logger.debug("RR EndPoint " + rrEndPoint);
							rrUser = accessPoint.username();
							logger.debug("RR User " + rrUser);
							rrPassword = StringEncrypter.getEncrypter()
									.decrypt(accessPoint.password());
							logger.debug("RR Password " + rrPassword);

							Map<String, Property> properties = accessPoint
									.propertyMap();

							Property property = properties.get("databaseName");
							rrDBName = property.name();
							logger.debug("RR DB Name " + rrDBName);
							rrDBDialect = properties.get("dialect").name();
							logger.debug("RR DB Dialect " + rrDBDialect);
							rrDBDriver = properties.get("driver").name();
							logger.debug("RR DB Driver " + rrDBDriver);

						}

					} catch (Exception e) {
						logger.error("RR " + dbType.getName() + " NOT FOUND", e);
					}
				}

			}
		}

		for (StatisticalType parameter : parameters) {

			logger.debug("Parameter retrieved " + parameter.getClass());
			if (parameter instanceof DatabaseType) {

				switch (((DatabaseType) parameter).getDatabaseParameter()) {
				case DATABASEURL:
					algoConfig.setParam(parameter.getName(),
							DataBaseManager.getUrlDB());
					break;
				case DATABASEPASSWORD:
					algoConfig.setParam(parameter.getName(),
							DataBaseManager.getPassword());
					break;
				case DATABASEUSERNAME:
					algoConfig.setParam(parameter.getName(),
							DataBaseManager.getUsername());
					break;
				case DATABASEDRIVER:
					algoConfig.setParam(
							parameter.getName(),
							(rrDBDriver != null) ? rrDBDriver : DataBaseManager
									.getDriver());
					break;
				case DATABASEDIALECT:
					if (rrDBDialect != null)
						algoConfig.setParam(parameter.getName(), rrDBDialect);

				case REMOTEDATABASEURL:
					algoConfig.setParam(parameter.getName(), rrEndPoint);
					break;

				case REMOTEDATABASEUSERNAME:
					algoConfig.setParam(parameter.getName(), rrUser);
					break;

				case REMOTEDATABASEPASSWORD:
					algoConfig.setParam(parameter.getName(), rrPassword);
					break;

				default:
					break;
				}
			}

			if (parameter instanceof ServiceType) {
				if (((ServiceType) parameter).getServiceParameter() == ServiceParameters.RANDOMSTRING) {
					String id = "ID_"
							+ UUID.randomUUID().toString().replace("-", "_");
					if (parameter.getDefaultValue() != null) {
						id = parameter.getDefaultValue() + id;
					}
					logger.debug("Param service name:" + parameter.getName()
							+ " value :" + id.toLowerCase());
					algoConfig.setParam(parameter.getName(), id.toLowerCase());

				}

				if (((ServiceType) parameter).getServiceParameter() == ServiceParameters.USERNAME) {
					logger.debug("Param service name:" + parameter.getName()
							+ " value :" + getID().getValue());
					algoConfig
							.setParam(parameter.getName(), getID().getValue());
				}

				// add code in order to obtaine generic runtime resource
				if (((ServiceType) parameter).getServiceParameter() == ServiceParameters.INFRA) {
					try {
						logger.debug("------ INFRA parameter ");
						String nameParam = parameter.getName();
						HashMap<String, String> generalProperties = new HashMap<String, String>();
						generalProperties = RuntimeResourceManager
								.getRRProfile(parameter.getName());
						for(Map.Entry<String, String> entry : generalProperties.entrySet())
						algoConfig.setParam(entry.getKey(), entry.getValue());

					} catch (Exception e) {
						logger.error("INFRA RR " + parameter.getName()
								+ " NOT FOUND", e);
					}
				}

			}

			if (parameter instanceof PrimitiveType) {
				if (((PrimitiveType) parameter).getType() == PrimitiveTypes.CONSTANT) {
					algoConfig.setParam(parameter.getName(),
							parameter.getDefaultValue());
					logger.debug("Param primitive name constant : "
							+ parameter.getName() + " value : "
							+ parameter.getDefaultValue());
				}
			}
		}
	}

	public void executeComputation(SMComputationConfig computationConfig,
			final long computationId) throws SMResourcesNotAvailableException {

		AlgorithmConfiguration algoConfig = new AlgorithmConfiguration();
		String algorithm = null;
		try {
			logger.debug(" ------------- Computation request: ");
			String configPath = getConfigPath();
			algorithm = computationConfig.getAlgorithm();
			logger.debug(" ------------- Algorithm request" + algorithm);

			algoConfig.setConfigPath(configPath);
			algoConfig.setAgent(algorithm);
			algoConfig.setModel(algorithm);
			algoConfig.setPersistencePath(configPath);

			logger.debug("Set user parameters init");
			setUserParameters(algoConfig, computationConfig);

			logger.debug("Set service paramter init");
			List<StatisticalType> parameters = getListParameters(computationConfig
					.getAlgorithm());
			setServiceParameters(computationConfig, algoConfig, parameters);
		} catch (Exception e) {
			logger.error(
					"Set service parameter error in execute computation :", e);
			try {
				SMPersistenceManager.setOperationStatus(computationId,
						SMOperationStatus.FAILED);
			} catch (Exception e1) {
				logger.error("Set status failed error in excute computation"
						+ " set parameters :", e1);
			}
			return;
		}

		try {
			logger.debug("Init computation");
			initComputation(computationConfig, algorithm, algoConfig,
					computationId);
		} catch (SMComputationalAgentInitializationException e) {
			logger.error("Init computation failed", e);
			try {
				SMPersistenceManager.setOperationStatus(computationId,
						SMOperationStatus.FAILED);
			} catch (Exception e1) {
				logger.error("Set status failed error in init computation :",
						e1);
			}
		}
	}

	private void startComputation(final long computationId,
			final INFRASTRUCTURE infra, final ComputationalAgent agent)
			throws SMComputationalAgentInitializationException {

		addComputationalAgent(String.valueOf(computationId), agent);

		new Thread() {
			@Override
			public void run() {
				try {
					SMPersistenceManager.setOperationStatus(computationId,
							SMOperationStatus.RUNNING);

					agent.init();
					agent.compute();

					setComputationOutput(computationId, agent);

				} catch (Exception e) {
					logger.error("Compute action failed", e);
					try {
						SMPersistenceManager.setOperationStatus(computationId,
								SMOperationStatus.FAILED);

					} catch (Exception e1) {
						logger.error("Error save status FAILED computation "
								+ computationId, e1);
					}

				} finally {
					cleanResourcesComputational(infra, computationId);
					removeComputationalAgent(String.valueOf(computationId));
				}
			}
		}.start();
	}

	private List<? extends ComputationalAgent> getComputationalAgentsAvailable(
			SMComputationConfig computationConfig,
			AlgorithmConfiguration algoConfig)
			throws SMComputationalAgentInitializationException {

		ComputationalAgentClass cac = null;
		try {
			cac = ComputationalAgentClass.fromString(ServiceUtil
					.getAlgorithmCategory(computationConfig.getAlgorithm()));
			switch (AlgorithmCategory.valueOf(cac.getValue())) {
			case DISTRIBUTIONS:
				return GeneratorsFactory.getGenerators(algoConfig);
			case EVALUATORS:
				return EvaluatorsFactory.getEvaluators(algoConfig);
			case CLUSTERERS:
				return ClusterersFactory.getClusterers(algoConfig);
			case MODELS:
				return ModelersFactory.getModelers(algoConfig);
			case TRANSDUCERS:
				return TransducerersFactory.getTransducerers(algoConfig);
			default:
				logger.error("Computational agent category not found ");
				throw new Exception();
			}
		} catch (Exception e) {
			logger.error("Computational agent list not found", e);
			throw new SMComputationalAgentInitializationException(
					"Algorithm requested filed : "
							+ ((cac != null) ? cac.getValue()
									: "computation agent not found"));
		}

	}

	private void initComputation(SMComputationConfig computationConfig,
			String algorithm, AlgorithmConfiguration algoConfig,
			long computationId)
			throws SMComputationalAgentInitializationException,
			SMResourcesNotAvailableException {

		logger.debug(" GET ComputationalAgent List ...");

		// TODO Ecological engine library throws an exception
		// if algoConfig has numberOfResources null so set temporarily 1 as
		// numberOfResources

		algoConfig.setNumberOfResources(1);

		List<? extends ComputationalAgent> agents = getComputationalAgentsAvailable(
				computationConfig, algoConfig);

		logger.debug(" FOUND Generetors List with size ..." + agents);
		for (ComputationalAgent agent : agents) {

			logger.debug("INFRASTRACTURE for ComputationalAgent found : "
					+ agent.getInfrastructure().toString());

			switch (agent.getInfrastructure()) {

			case D4SCIENCE: {

				logger.debug("Start D4Science computation");

				if (!ComputationFactory.getFactoryResource()
						.setD4ScienceComputation()) {
					logger.debug("D4Science resource not available");
					break;
				}

				try {

					SMPersistenceManager.setComputationalInfrastructure(
							computationId, INFRASTRUCTURE.D4SCIENCE);

					String infra = (String) GHNContext.getContext()
							.getProperty(GHNContext.INFRASTRUCTURE_NAME, true);

					algoConfig.setGcubeScope("/" + infra);

					logger.debug("Retrieve ComputationalAgent parameters .... ");
					List<StatisticalType> parameters = agent
							.getInputParameters();
					setServiceParameters(computationConfig, algoConfig,
							parameters);
					agent.setConfiguration(algoConfig);
					startComputation(computationId, INFRASTRUCTURE.D4SCIENCE,
							agent);
					logger.debug("Computation started in D4Science ... ");

					return;
				} catch (Exception e) {

					// Clean monitor resources
					logger.error("Start computation failed", e);
					cleanResourcesComputational(INFRASTRUCTURE.D4SCIENCE,
							computationId);
					throw new SMComputationalAgentInitializationException(
							"ComputationalAgent initialization failed "
									+ e.getMessage());
				}

			}

			case LOCAL: {

				int resources = 0;
				try {
					// resources = ComputationFactory.getFactoryResource().
					// setLocalResourcesAvailable(String.valueOf(computationId),
					// computationConfig.getAlgorithm(),
					// ServiceUtil.getAlgorithmCategory(computationConfig.getAlgorithm()));
					resources = ComputationFactory
							.getFactoryResource()
							.getLocalResourcesNeeded(
									computationConfig.getAlgorithm(),
									ServiceUtil
											.getAlgorithmCategory(computationConfig
													.getAlgorithm()));
					logger.debug("---------> Resources needed :" + resources);

				} catch (Exception e) {
					throw new SMComputationalAgentInitializationException(
							"ComputationalAgent initialization failed "
									+ e.getMessage());
				}

				// if (resources <= 0)
				// throw new
				// SMResourcesNotAvailableException("Local Resources not available");
				boolean free = ComputationFactory.getFactoryResource()
						.getLocalComputation();
				logger.debug("---------> Resources free   :" + free);
				if (!free)
					throw new SMResourcesNotAvailableException(
							"Local Resources not available");

				try {

					logger.debug("Computation started in LOCAL ...");
					SMPersistenceManager.setComputationalInfrastructure(
							computationId, INFRASTRUCTURE.LOCAL);

					logger.debug("Set number of resources ");
					algoConfig.setNumberOfResources(resources);
					agent.setConfiguration(algoConfig);
					startComputation(computationId, INFRASTRUCTURE.LOCAL, agent);

					return;
				} catch (Exception e) {
					// Clean monitor resources
					logger.error("Start computation failed", e);
					cleanResourcesComputational(INFRASTRUCTURE.LOCAL,
							computationId);
					throw new SMComputationalAgentInitializationException(
							"ComputationalAgent initialization failed "
									+ e.getMessage());
				}

			}
			}
		}

		throw new SMComputationalAgentInitializationException(
				"Computation initialization failed");

	}
}
