package org.gcube.data.analysis.statisticalmanager.persistence;

import static org.gcube.data.analysis.statisticalmanager.persistence.HibernateManager.closeSession;
import static org.gcube.data.analysis.statisticalmanager.persistence.HibernateManager.getSessionFactory;
import static org.gcube.data.analysis.statisticalmanager.persistence.HibernateManager.roolbackTransaction;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.UUID;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.analysis.statisticalmanager.SMOperationStatus;
import org.gcube.data.analysis.statisticalmanager.SMOperationType;
import org.gcube.data.analysis.statisticalmanager.SMResourceType;
import org.gcube.data.analysis.statisticalmanager.stubs.SMComputationRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMComputations;
import org.gcube.data.analysis.statisticalmanager.stubs.SMCreateTableRequest;
import org.gcube.data.analysis.statisticalmanager.stubs.SMImporters;
import org.gcube.data.analysis.statisticalmanager.stubs.SMResources;
import org.gcube.dataanalysis.ecoengine.configuration.INFRASTRUCTURE;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.TableTemplates;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMAbstractResource;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMComputation;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMEntries;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMEntry;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMError;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMFile;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMImport;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMInputEntry;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMObject;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMOperation;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMResource;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMSystemImport;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMTable;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.transform.Transformers;

public class SMPersistenceManager {

	private  static GCUBELog logger = new GCUBELog(SMPersistenceManager.class);

	public static long addImporter(final SMCreateTableRequest request) throws Exception {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			
			SMImport smimport = new SMImport();
			smimport.setFileName(request.getTableName());
			smimport.setOperationType(SMOperationType.IMPORTED.ordinal());
			smimport.setPortalLogin(request.getUser());
			smimport.setSubmissionDate(Calendar.getInstance());
			smimport.setDescription(request.getDescription());
			smimport.setOperationStatus(SMOperationStatus.RUNNING.ordinal());
			session.save(smimport);
			t.commit();
			return smimport.getOperationId();

		} catch (Exception e) {
			roolbackTransaction(t);
			throw new Exception();
		} finally {
			closeSession(session);
		}
	}
	
	public static void addCreatedResource(SMResource resource) throws Exception {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			if (session.get(SMResource.class, resource.getResourceId()) == null) 
				session.save(resource);
			
			t.commit();
		} catch (Exception e) {
			roolbackTransaction(t);
			throw new Exception();
		} finally {
			closeSession(session);
		}
	}
	
	public static long addSystemImporter(String description, SMResource resource) throws Exception {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {

			SMSystemImport smimport = new SMSystemImport();
			smimport.setOperationType(SMOperationType.SYSTEM.ordinal());
			smimport.setSubmissionDate(Calendar.getInstance());
			smimport.setDescription(description);
			session.save(resource);
			
			SMAbstractResource ar = new SMAbstractResource();
			ar.setResource(resource);
			ar.setAbstractResourceId(resource.getResourceId());
			session.save(ar);

			smimport.setAbstractResource(ar);
			smimport.setOperationStatus(SMOperationStatus.COMPLETED.ordinal());
			smimport.setCompletedDate(Calendar.getInstance());
			session.saveOrUpdate(smimport);
			
			t.commit();
			return smimport.getOperationId();
		} catch (Exception e) {
			roolbackTransaction(t);
			throw new Exception();
		} finally {
			closeSession(session);
		}
	}
	 	
	public static List<SMResource> getResources(String user, String template) {

		Session session = HibernateManager.getSessionFactory().openSession();
		try {
			Query query = session
			.createQuery("select resource "
					+ "from SMResource resource " 
					+ "where (resource.portalLogin like :name or resource.portalLogin = null) "
					+ "and resource.resourceType <> 2 "
					+ "and resource.resourceType <> 3 ");

			query.setParameter("name",
					(user != null) ? user : "%");
			
			return query.list();

		} finally {
			session.close();
		}
	}


	public static SMOperation getOperation(long operationId) throws Exception{

		Session session = HibernateManager.getSessionFactory().openSession();
		try {
			return (SMOperation)session.get(SMOperation.class, operationId);
		} finally {
			closeSession(session);
		}
	}

	public static SMComputations getComputations(final String user,
			final String algorithm, final String category) throws Exception{

		Session session = HibernateManager.getSessionFactory().openSession();
		try {
			Query query = session
			.createQuery("select computation from SMComputation  computation "
					+ "where computation.portalLogin like :name and "
					+ "computation.algorithm like :algorithm and "
					+ "computation.category like :category");

			query.setParameter("name", (user !=null)?user:"%");
			query.setParameter("algorithm", (algorithm !=null)?algorithm:"%");
			query.setParameter("category", (category !=null)?category:"%");

			@SuppressWarnings("unchecked")
			List<Object> objects = query.list();

			for(Object object : objects) {
				SMComputation computation = (SMComputation)object;

				Query queryParameters = session.createQuery(
						"select parameter from SMEntry parameter " +
				"where parameter.computationId = :computationId");
				queryParameters.setParameter("computationId", computation.getOperationId());

				@SuppressWarnings("unchecked")
				List<Object> parameters = queryParameters.list();
				if (!parameters.isEmpty()) {
					computation.setParameters(parameters.toArray(new SMEntry[parameters.size()]));
				}		
			}

			SMComputation[] computations = objects.toArray(new SMComputation[objects.size()]);
			return new SMComputations(computations);

		} finally {
			closeSession(session);
		}
	}

	public static long addComputation(final SMComputationRequest request, String category) throws Exception {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {

			SMComputation smcomputation = new SMComputation();
			smcomputation.setOperationType(SMOperationType.COMPUTED.ordinal());
			smcomputation.setPortalLogin(request.getUser());
			smcomputation.setSubmissionDate(Calendar.getInstance());
			smcomputation.setTitle(request.getTitle());
			smcomputation.setDescription(request.getDescription());

			String algorithm = request.getConfig().getAlgorithm();
			smcomputation.setAlgorithm(algorithm);
			smcomputation.setCategory(category);

			smcomputation.setOperationStatus(SMOperationStatus.PENDING.ordinal());
			session.save(smcomputation);

			SMEntries parameters = request.getConfig().getParameters();
			
			if (parameters.getList() != null) 
				for(SMInputEntry parameter : parameters.getList()) {
					SMEntry entry = new SMEntry();
					entry.setKey(parameter.getKey());
					entry.setValue(parameter.getValue());
					entry.setComputationId(smcomputation.getOperationId());
					session.save(entry);
				}

			t.commit();
			return smcomputation.getOperationId();

		} catch (Exception e) {
			roolbackTransaction(t);
			e.printStackTrace();
			throw new Exception();
		} finally {
			closeSession(session);
		}
	}

	public static void addCreatedResource(
			final long operationId, SMResource resource) throws Exception {

		logger.debug("------------------------------------------------------");
		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {

			SMOperation operation = (SMOperation)session.get(SMOperation.class, operationId);
			logger.debug("Resource type " + SMResourceType.values()[resource.getResourceType()]);
			switch (SMResourceType.values()[resource.getResourceType()]) {
			
//			case TABULAR:
//				SMTable table = (SMTable)resource;
//				session.save(table);
//				break;
//			case FILE:
//				SMFile file = (SMFile)resource;
//				session.save(file);
//				break;
			case OBJECT:
				
				SMObject object = (SMObject)resource;
				
				session.save(object);
				logger.debug("Resource saved !!!!");
				logger.debug("-----------------------------------------------------------");
				break;
			}
			

			SMAbstractResource ar = new SMAbstractResource();
			ar.setResource(resource);
			ar.setAbstractResourceId(resource.getResourceId());
			session.save(ar);

			operation.setAbstractResource(ar);
			operation.setOperationStatus(SMOperationStatus.COMPLETED.ordinal());
			operation.setCompletedDate(Calendar.getInstance());
			session.saveOrUpdate(operation);
			t.commit();
		} catch (Exception e) {
			logger.debug("Error persistence ", e);
			roolbackTransaction(t);
			throw new Exception();
		} finally {
			closeSession(session);
		}
	}

	public static void setOperationStatus(long operationId, SMOperationStatus status) {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			SMOperation smoperation = (SMOperation)session.get(SMOperation.class, operationId);
			smoperation.setOperationStatus(status.ordinal());

			if (status == SMOperationStatus.FAILED) {
				smoperation.setCompletedDate(Calendar.getInstance());
				
				SMError error = new SMError("General error");
				error.setDescription("General Error");
				
				error.setResourceId(UUID.randomUUID().toString());
				error.setResourceType(SMResourceType.ERROR.ordinal());
				
				session.save(error);
				
				SMAbstractResource ar = new SMAbstractResource();
				ar.setResource(error);
				ar.setAbstractResourceId(error.getResourceId());
				session.save(ar);
				
				smoperation.setAbstractResource(ar);
				
			}
			
			if (status == SMOperationStatus.PENDING)
				smoperation.setSubmissionDate(Calendar.getInstance());

			session.saveOrUpdate(smoperation);
			t.commit();		
		} catch (Exception e) {
			roolbackTransaction(t);
		} finally {
			closeSession(session);
		}
	}

	public static void setComputationalInfrastructure(long operationId, INFRASTRUCTURE infra) {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			SMComputation smcomputation = (SMComputation)session.get(SMComputation.class, operationId);
			smcomputation.setInfrastructure(infra.toString());
			session.save(smcomputation);
			t.commit();		
		} catch (Exception e) {
			roolbackTransaction(t);
		} finally {
			closeSession(session);
		}
	}
	
	private static void removeResources(Session session,long operationId) {
		
		Query query = session
				.createQuery("select resource from SMResource  resource "
						+ "where resource.operationId = :operationId ");
		
		query.setParameter("operationId", operationId);
		
		List<SMResource> resources =  query.list();
		if (resources != null) {
			for (SMResource resource : resources) {

				switch (SMResourceType.values()[resource.getResourceType()]) {
				case TABULAR:
					try {
						DataBaseManager.removeTable(resource.getResourceId());
					} catch (Exception e) {
						logger.error("Table no deleted", e);
					}
					break;
				case FILE:
					break;

				case OBJECT:
					break;

				default:
					break;
				}
				session.delete(resource);
			}
		}
		
	}

	public static void removeComputation(long operationId) {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			
			SMComputation smoperation = (SMComputation)session.get(SMComputation.class, operationId);
			removeResources(session, operationId);
			
			if (smoperation != null)
				session.delete(smoperation);
			t.commit();		
		} catch (Exception e) {
			logger.error("Computation not removed ",e);
			roolbackTransaction(t);
		} finally {
			closeSession(session);
		}
	}

	public static void removeResource(String resourceId) {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			SMResource resource = (SMResource)session.get(SMResource.class, resourceId);
			if (resource != null) {
				SMOperation operation = (SMOperation)session.get(SMOperation.class, resource.getOperationId());

				if (operation != null)
					session.delete(operation);
				
				removeResources(session, resource.getOperationId());
			}
			
			t.commit();		
		} catch (Exception e) {
			logger.error("Computation not removed ",e);
			roolbackTransaction(t);
		} finally {
			closeSession(session);
		}
	}
	
	public static void removeImporter(long importerId) {

		Session session = getSessionFactory().openSession();
		Transaction t = session.beginTransaction();
		try {
			SMOperation operation = (SMOperation)session.get(SMOperation.class, importerId);
			if (operation != null)
				session.delete(operation);
			t.commit();		
		} catch (Exception e) {
			logger.error("Computation not removed ",e);
			roolbackTransaction(t);
		} finally {
			closeSession(session);
		}
		
	}


	
	public static SMFile getFile(String fileId) {

		Session session = getSessionFactory().openSession();
		try {
			Query query = session
			.createQuery("select file from SMFile  file "
					+ "where resourceId = :fileId");

			query.setParameter("fileId",fileId);
			
			@SuppressWarnings("unchecked")
			List<Object> objects = query.list();
			if(!objects.isEmpty())
				return (SMFile)objects.get(0);
			else
				return null;
			
		} finally {
			closeSession(session);
		}
	}

}
