package org.gcube.data.analysis.excel.data.loader.impl;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.gcube.data.analysis.excel.data.CellWrapper;
import org.gcube.data.analysis.excel.data.TimeSeriesColumn;
import org.gcube.data.analysis.excel.data.TimeSeriesColumn.DATA_TYPE;
import org.gcube.data.analysis.excel.data.TimeSeriesTable;
import org.gcube.data.analysis.excel.data.exceptions.TimeSeriesLoaderException;
import org.gcube.data.analysis.excel.data.factory.CellWrapperFactory;
import org.gcube.data.analysis.excel.data.factory.TimeSeriesTableFactory;
import org.gcube.data.analysis.excel.data.impl.BlankCellWrapper;
import org.gcube.data.analysis.excel.data.impl.GenericCellWrapper;
import org.gcube.data.analysis.excel.data.impl.TimeSeriesColumnImpl;
import org.gcube.data.analysis.excel.data.loader.TimeSeriesTableLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExcelTimeSeriesLoader implements TimeSeriesTableLoader {

	private Logger logger;

	private TimeSeriesTable table;
	
	public ExcelTimeSeriesLoader() {
		this.logger = LoggerFactory.getLogger(this.getClass());
		this.table = null;
	}
	
	
	
	@Override
	public void initTable(InputStream stream) throws TimeSeriesLoaderException {
		
		try
		{	
			this.logger.debug("Loading file stream");
			XSSFWorkbook document = new XSSFWorkbook(stream);
			Sheet sheet = document.getSheetAt(0);
			List<List<CellWrapper>> cells = new ArrayList<>();
			String name = sheet.getSheetName();
			this.logger.debug("Table name "+name);
			Iterator<Row> rows =  sheet.iterator();
			int nColumns =0;
			
			if (rows.hasNext())
			{
				this.logger.debug("Parsing headers");
				Row firstRow = rows.next();
				Iterator<Cell> firstRowCells = firstRow.iterator();
				
				while (firstRowCells.hasNext())
				{
					Cell cell = firstRowCells.next();
					nColumns ++;
					List<CellWrapper> cellList = new ArrayList<>();
					cellList.add(new GenericCellWrapper(DATA_TYPE.GENERIC,cell.getStringCellValue()));
					cells.add(cellList);
					this.logger.debug("Header cell added");
				}

			}
			else
			{	document.close();
				throw new TimeSeriesLoaderException("Invalid excel table: no header");
			}
			
			while (rows.hasNext())
			{
				int columnIndex = 0;
				Iterator<Cell> rowCells = rows.next().iterator();
				
				while (rowCells.hasNext())
				{
	
					this.logger.debug("Adding data row");
					Cell cell = rowCells.next();
					int currentColumnIndex = cell.getColumnIndex();
					this.logger.debug("Current column index "+columnIndex);
					
					
					if (currentColumnIndex >= nColumns)
					{
						this.logger.warn("Invalid column: no header present");
					}
					else 
					{
						for (; columnIndex < currentColumnIndex; columnIndex++)
						{
							this.logger.debug("adding an empty cell");
							cells.get(columnIndex).add(new BlankCellWrapper());
						}

						this.logger.debug("Adding cell "+cell.getCellTypeEnum());
						cells.get(columnIndex).add(CellWrapperFactory.generateCellWrapper(cell));

					}
					columnIndex ++;
					this.logger.debug("Data row added");
				}
				
				
			}
			
			document.close();
			
			Set<TimeSeriesColumn> timeSeriesColumns = new HashSet<>();
			
			for (List<CellWrapper> column : cells)
			{
				timeSeriesColumns.add(new TimeSeriesColumnImpl(column));
			}

			
			this.table = TimeSeriesTableFactory.generateTimeSeriesTable(name, timeSeriesColumns);
			this.logger.debug("Table created");
			
		} catch (Exception e)
		{
			throw new TimeSeriesLoaderException("Unable to load excel file", e);
		}
		


	}
	
	
//	private List<Cell> parseListCell (List<Cell> cells)
//	{
//		List<Cell> response = new ArrayList<>(cells);
//		
//		if (response.size()>1)
//		{
//			this.logger.debug("Verifying data type of the data cells");
//			
//			boolean found = false;
//			Iterator<Cell> cellIterator = response.iterator();
//			cellIterator.next();
//			
//			while (cellIterator.hasNext() && ! found)
//			{
//				Cell cell = cellIterator.next();
//				if (cell.getCellTypeEnum() != CellType.BLANK && (cell.getCellTypeEnum() != CellType.STRING && cell.getStringCellValue().trim().length() != 0)) found = true;
//			}
//			
//			if (!found)
//			{
//				response.clear();
//				response.add(cells.get(0));
//			}
//		}
//		return response;
//	}

	@Override
	public TimeSeriesTable getTimeSeries() 
	{

		return this.table;
	}

}
