package org.gcube.data.analysis.cmemsimportercl.server.cl;

import java.io.IOException;
import java.util.ArrayList;

import javax.ws.rs.client.ClientRequestContext;
import javax.ws.rs.client.ClientRequestFilter;

import org.gcube.data.analysis.cmemsimportercl.server.cl.command.products.CCmdCMEMSDatasetInfo;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.products.CCmdCMEMSDatasets;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.products.CCmdCMEMSKeywords;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.products.CCmdCMEMSMotus;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.products.CCmdCMEMSProducts;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.products.CCmdCMEMSRegions;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.products.CCmdCMEMSTimeLimits;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.products.CCmdCMEMSVariables;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.tasks.CCmdCMEMSExecuteTask;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.tasks.CCmdCMEMSExecutionInfo;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.tasks.CCmdCMEMSExecutions;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.tasks.CCmdCMEMSPreviewChunk;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.tasks.CCmdCMEMSPreviewSize;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.tasks.CCmdCMEMSReports;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.tasks.CCmdCMEMSScheduleTask;
import org.gcube.data.analysis.cmemsimportercl.server.cl.command.tasks.CCmdCMEMSTasks;
import org.gcube.data.analysis.cmemsimportercl.server.is.InformationSystemUtils;
import org.gcube.data.analysis.cmemsimportercl.server.util.ServiceCredentials;
import org.gcube.data.analysis.cmemsimportercl.server.util.ServiceSupportData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.products.DatasetData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.products.DatasetInfoData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.products.DatasetType;
import org.gcube.data.analysis.cmemsimportercl.shared.data.products.KeywordData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.products.MotuData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.products.ProductData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.products.RegionData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.products.TimeLimitsData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.products.VariableData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.tasks.ExecutionData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.tasks.ExecutionReportData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.tasks.ImportOptionsData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.tasks.ImportTaskData;
import org.gcube.data.analysis.cmemsimportercl.shared.exception.ServiceException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * CClient 4 CMEMS Service
 * 
 * 
 * @author Giancarlo Panichi
 *
 *
 */
public class CClient4CMEMSImporterService extends CClient {

	private static final long serialVersionUID = 1326328215386650188L;
	private static Logger logger = LoggerFactory.getLogger(CClient4CMEMSImporterService.class);

	private ServiceSupportData serviceSupportData;

	/**
	 * 
	 * @param serviceCredentials
	 *            Service Credentials
	 * @throws ServiceException
	 *             Service Exception
	 */
	public CClient4CMEMSImporterService(ServiceCredentials serviceCredentials) throws ServiceException {
		super();
		if (serviceCredentials == null) {
			logger.error("Error credetials are null!");
			throw new ServiceException("Error credetials are null!");
		}

		if (serviceCredentials.getToken() == null || serviceCredentials.getToken().isEmpty()) {
			logger.error("Error authorization token invalid: " + serviceCredentials.getToken());
			throw new ServiceException("Error authorization token invalid: " + serviceCredentials.getToken());
		}

		if (serviceCredentials.getUserName() == null || serviceCredentials.getUserName().isEmpty()) {
			logger.error("Error invalid user name: " + serviceCredentials.getUserName());
			throw new ServiceException("Error invalid user name: " + serviceCredentials.getUserName());
		}

		String serviceAddress;
		try {
			serviceAddress = InformationSystemUtils.retrieveCMEMSImporterService(serviceCredentials.getScope());
		} catch (Exception e) {
			logger.error("Error retrieving service address: " + e.getLocalizedMessage());
			e.printStackTrace();
			throw new ServiceException(e.getLocalizedMessage(), e);
		}
		logger.debug("Service Address retrieved: " + serviceAddress);
		if (serviceAddress == null || serviceAddress.isEmpty()) {
			logger.error("CMEMS Importer Service address not available!");
			throw new ServiceException("CMEMS Importer Service address not available!");
		}

		ClientRequestFilter authFilter = getAuthFilter(serviceCredentials.getToken());

		serviceSupportData = new ServiceSupportData(serviceCredentials, serviceAddress, authFilter);

	}

	/**
	 * 
	 * @param serviceCredentials
	 *            Service Credentials
	 * @param serviceAddressUrl
	 *            Service Address Url
	 * @throws ServiceException
	 *             Service Exception
	 */
	public CClient4CMEMSImporterService(ServiceCredentials serviceCredentials, String serviceAddressUrl)
			throws ServiceException {
		super();
		if (serviceCredentials == null) {
			logger.error("Error credetials are null!");
			throw new ServiceException("Error credetials are null!");
		}

		if (serviceCredentials.getToken() == null || serviceCredentials.getToken().isEmpty()) {
			logger.error("Error authorization token invalid: " + serviceCredentials.getToken());
			throw new ServiceException("Error authorization token invalid: " + serviceCredentials.getToken());
		}

		if (serviceCredentials.getUserName() == null || serviceCredentials.getUserName().isEmpty()) {
			logger.error("Error invalid user name: " + serviceCredentials.getUserName());
			throw new ServiceException("Error invalid user name: " + serviceCredentials.getUserName());
		}

		logger.debug("Service Address retrieved: " + serviceAddressUrl);
		if (serviceAddressUrl == null || serviceAddressUrl.isEmpty()) {
			logger.error("CMEMS Importer Service address not available!");
			throw new ServiceException("CMEMS Importer Service address not available!");
		}

		ClientRequestFilter authFilter = getAuthFilter(serviceCredentials.getToken());

		serviceSupportData = new ServiceSupportData(serviceCredentials, serviceAddressUrl, authFilter);

	}

	private ClientRequestFilter getAuthFilter(String token) {
		ClientRequestFilter authFilter = new ClientRequestFilter() {
			public void filter(ClientRequestContext requestContext) throws IOException {
				requestContext.getHeaders().add("gcube-token", token);
			}
		};
		return authFilter;
	}

	@Override
	public String toString() {
		return "CMEMS Importer Client";
	}

	@Override
	public ArrayList<ProductData> getProducts() throws Exception {
		CCmdCMEMSProducts cCmdCMEMSProducts = new CCmdCMEMSProducts(serviceSupportData);
		ArrayList<ProductData> result = cCmdCMEMSProducts.execute();
		return result;
	}

	@Override
	public ArrayList<DatasetData> getDatasets(String productId) throws Exception {
		CCmdCMEMSDatasets cCmdCMEMSDatasets = new CCmdCMEMSDatasets(serviceSupportData, productId);
		ArrayList<DatasetData> result = cCmdCMEMSDatasets.execute();
		return result;
	}

	@Override
	public DatasetInfoData getDatasetInfo(String productId, DatasetType datasetType, String datasetId)
			throws Exception {
		CCmdCMEMSDatasetInfo cCmdCMEMSDatasetInfo = new CCmdCMEMSDatasetInfo(serviceSupportData, productId, datasetType,
				datasetId);
		DatasetInfoData result = cCmdCMEMSDatasetInfo.execute();
		return result;
	}

	@Override
	public ArrayList<MotuData> getMotus() throws Exception {
		CCmdCMEMSMotus cCmdCMEMSMotus = new CCmdCMEMSMotus(serviceSupportData);
		ArrayList<MotuData> result = cCmdCMEMSMotus.execute();
		return result;
	}

	@Override
	public ArrayList<KeywordData> getKeywords() throws Exception {
		CCmdCMEMSKeywords cCmdCMEMSKeyword = new CCmdCMEMSKeywords(serviceSupportData);
		ArrayList<KeywordData> result = cCmdCMEMSKeyword.execute();
		return result;
	}

	@Override
	public ArrayList<RegionData> getRegions() throws Exception {
		CCmdCMEMSRegions cCmdCMEMSRegions = new CCmdCMEMSRegions(serviceSupportData);
		ArrayList<RegionData> result = cCmdCMEMSRegions.execute();
		return result;
	}

	@Override
	public ArrayList<VariableData> getVariables() throws Exception {
		CCmdCMEMSVariables cCmdCMEMSVariables = new CCmdCMEMSVariables(serviceSupportData);
		ArrayList<VariableData> result = cCmdCMEMSVariables.execute();
		return result;
	}

	@Override
	public TimeLimitsData getTimeLimits() throws Exception {
		CCmdCMEMSTimeLimits cCmdCMEMSTime = new CCmdCMEMSTimeLimits(serviceSupportData);
		TimeLimitsData result = cCmdCMEMSTime.execute();
		return result;
	}

	@Override
	public ArrayList<ImportTaskData> getTasks() throws Exception {
		CCmdCMEMSTasks cCmdCMEMSTasks = new CCmdCMEMSTasks(serviceSupportData);
		ArrayList<ImportTaskData> result = cCmdCMEMSTasks.execute();
		return result;
	}

	@Override
	public ArrayList<ExecutionData> getExecutions(String taskId) throws Exception {
		CCmdCMEMSExecutions cCmdCMEMSExecutions = new CCmdCMEMSExecutions(serviceSupportData, taskId);
		ArrayList<ExecutionData> result = cCmdCMEMSExecutions.execute();
		return result;
	}

	@Override
	public ArrayList<ExecutionReportData> getReports(String taskId, String executionId) throws Exception {
		CCmdCMEMSReports cCmdCMEMSReports = new CCmdCMEMSReports(serviceSupportData, taskId, executionId);
		ArrayList<ExecutionReportData> result = cCmdCMEMSReports.execute();
		return result;
	}

	@Override
	public String getPreviewSize(ImportOptionsData importOptionsData) throws Exception {
		CCmdCMEMSPreviewSize cCmdCMEMSPreviewSize = new CCmdCMEMSPreviewSize(serviceSupportData, importOptionsData);
		String result = cCmdCMEMSPreviewSize.execute();
		return result;
	}

	@Override
	public String getPreviewChunk(ImportOptionsData importOptionsData) throws Exception {
		CCmdCMEMSPreviewChunk cCmdCMEMSPreviewChunk = new CCmdCMEMSPreviewChunk(serviceSupportData, importOptionsData);
		String result = cCmdCMEMSPreviewChunk.execute();
		return result;
	}

	@Override
	public ExecutionData executeTask(String taskId) throws Exception {
		CCmdCMEMSExecuteTask cCmdCMEMSExecuteTask = new CCmdCMEMSExecuteTask(serviceSupportData, taskId);
		ExecutionData result = cCmdCMEMSExecuteTask.execute();
		return result;
	}

	@Override
	public ImportTaskData scheduleTask(ImportOptionsData importOptionsData) throws Exception {
		CCmdCMEMSScheduleTask cCmdCMEMSScheduleTask = new CCmdCMEMSScheduleTask(serviceSupportData, importOptionsData);
		ImportTaskData result = cCmdCMEMSScheduleTask.execute();
		return result;
	}

	@Override
	public ExecutionData executionInfo(String taskId, String executionId) throws Exception {
		CCmdCMEMSExecutionInfo cCmdCMEMSExecutecutionInfo = new CCmdCMEMSExecutionInfo(serviceSupportData, taskId,
				executionId);
		ExecutionData result = cCmdCMEMSExecutecutionInfo.execute();
		return result;
	}

}
