package org.gcube.data.analysis.cmemsimportercl.server.cl.command.tasks;

import java.util.ArrayList;
import java.util.Calendar;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.gcube.data.analysis.cmemsimportercl.server.cl.command.CCmd;
import org.gcube.data.analysis.cmemsimportercl.server.util.ServiceSupportData;
import org.gcube.data.analysis.cmemsimportercl.server.util.TimeUtils;
import org.gcube.data.analysis.cmemsimportercl.shared.Constants;
import org.gcube.data.analysis.cmemsimportercl.shared.data.tasks.ExecutionData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.tasks.ExecutionReportData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.tasks.ImportOptionsData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.tasks.ImportTaskData;
import org.gcube.data.analysis.cmemsimportercl.shared.data.tasks.SubmissionInfoData;
import org.gcube.data.analysis.cmemsimportercl.shared.exception.ServiceException;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ChunkTimespan;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ExecutionReport;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ImportTask;
import org.glassfish.jersey.client.ClientConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * @author Giancarlo Panichi
 *
 *
 */
public class CCmdCMEMSScheduleTask implements CCmd<ImportTaskData> {
	private static final Logger logger = LoggerFactory.getLogger(CCmdCMEMSScheduleTask.class);

	private ServiceSupportData serviceSupportData;
	private ImportOptionsData importOptionsData;

	public CCmdCMEMSScheduleTask(ServiceSupportData serviceSupportData, ImportOptionsData importOptionsData) {
		this.serviceSupportData = serviceSupportData;
		this.importOptionsData = importOptionsData;
		logger.debug("CCmdCMEMSScheduleTask");
		logger.debug("ImportOptionsData: " + importOptionsData);
	}

	@Override
	public ImportTaskData execute() throws ServiceException {
		try {
			ClientConfig config = new ClientConfig();
			config.register(serviceSupportData.getAuthFilter());
			Client client = ClientBuilder.newClient(config);
			WebTarget service = client.target(serviceSupportData.getServiceAddress());

			service = service.path("tasks");
			logger.debug("Request URI: " + service.getUri().toString());

			ImportOptions importOptions = new ImportOptions();
			importOptions.setProduct(importOptionsData.getProduct());
			importOptions.setDataset(importOptionsData.getDataset());
			importOptions.setMotu(importOptionsData.getMotu());
			importOptions.setVariables(importOptionsData.getVariables());
			importOptions.setBackTime(importOptionsData.getBackTime());

			if (importOptionsData.getChunkSpan() != null && !importOptionsData.getChunkSpan().isEmpty()) {
				
				logger.debug("ChunkTimespan: "+ChunkTimespan.YEAR.toString());
				ChunkTimespan chunkTimeSpan = ChunkTimespan.fromString(importOptionsData.getChunkSpan());
				importOptions.setChunkSpan(chunkTimeSpan);
			}

			if (importOptionsData.gettLo() != null) {
				importOptions.settLo(TimeUtils.toCalendar(importOptionsData.gettLo()));
			}
			if (importOptionsData.gettHi() != null) {
				importOptions.settHi(TimeUtils.toCalendar(importOptionsData.gettHi()));
			}
			if (importOptionsData.getxLo() != null) {
				importOptions.setxLo(Double.valueOf(importOptionsData.getxLo()));
			}
			if (importOptionsData.getxHi() != null) {
				importOptions.setxHi(Double.valueOf(importOptionsData.getxHi()));
			}

			if (importOptionsData.getyLo() != null) {
				importOptions.setyLo(Double.valueOf(importOptionsData.getyLo()));
			}
			if (importOptionsData.getyHi() != null) {
				importOptions.setyHi(Double.valueOf(importOptionsData.getyHi()));
			}
			if (importOptionsData.getzLo() != null) {
				importOptions.setzLo(Double.valueOf(importOptionsData.getzLo()));
			}
			if (importOptionsData.getzHi() != null) {
				importOptions.setzHi(Double.valueOf(importOptionsData.getzHi()));
			}

			if (importOptionsData.getImportSchedule() != null && !importOptionsData.getImportSchedule().isEmpty()) {
				importOptions.setImportSchedule(importOptionsData.getImportSchedule());
			} else {
				importOptions.setImportSchedule(Constants.CMEMS_IMPORTER_SERVICE_DEFAULT_RUN);
			}
			
			Entity<ImportOptions> data = Entity.entity(importOptions, MediaType.APPLICATION_XML);

			Response response = service.request(MediaType.APPLICATION_XML).accept(MediaType.APPLICATION_XML).post(data);

			logger.debug("Response: [" + response + "]");

			ImportTask importTask = response.readEntity(ImportTask.class);

			logger.debug("ImportTask: [" + importTask + "]");

			ImportTaskData taskData = null;

			if (importTask != null) {

				SubmissionInfoData submissionInfoData = new SubmissionInfoData();

				if (importTask.getSubmissionInfo() != null) {

					String timeScheduled = "";
					try {
						Calendar calendarScheduled = importTask.getSubmissionInfo().getScheduled();
						if (calendarScheduled != null) {
							timeScheduled = TimeUtils.toString(calendarScheduled);
						}
					} catch (Throwable e) {
						logger.error(e.getLocalizedMessage(), e);
					}

					submissionInfoData.setUser(importTask.getSubmissionInfo().getUser());
					submissionInfoData.setScope(importTask.getSubmissionInfo().getScope());
					submissionInfoData.setToken(importTask.getSubmissionInfo().getToken());
					submissionInfoData.setScheduled(timeScheduled);

				}

				ImportOptionsData importOptionData = new ImportOptionsData();

				if (importTask.getImportParameters() != null) {

					String timeTLo = "";
					try {
						Calendar calendarTLo = importTask.getImportParameters().gettLo();
						if (calendarTLo != null) {
							timeTLo = TimeUtils.toString(calendarTLo);
						}
					} catch (Throwable e) {
						logger.error(e.getLocalizedMessage(), e);
					}

					String timeTHi = "";
					try {
						Calendar calendarTHi = importTask.getImportParameters().gettHi();
						if (calendarTHi != null) {
							timeTHi = TimeUtils.toString(calendarTHi);
						}
					} catch (Throwable e) {
						logger.error(e.getLocalizedMessage(), e);
					}

					int backTime = 0;
					if (importTask.getImportParameters().getBackTime() != null) {
						backTime = importTask.getImportParameters().getBackTime();
					}

					importOptionData.setBackTime(backTime);
					importOptionData.setDataset(importTask.getImportParameters().getDataset());
					importOptionData.setImportSchedule(importTask.getImportParameters().getImportSchedule());
					importOptionData.setMotu(importTask.getImportParameters().getMotu());
					importOptionData.setProduct(importTask.getImportParameters().getProduct());

					if (importTask.getImportParameters().getChunkSpan() != null) {
						importOptionData.setChunkSpan(importTask.getImportParameters().getChunkSpan().toString());
					}

					importOptionData.settLo(timeTLo);
					importOptionData.settHi(timeTHi);

					ArrayList<String> variables = new ArrayList<>();

					if (importTask.getImportParameters().getVariables() != null) {
						variables.addAll(importTask.getImportParameters().getVariables());
					}
					importOptionData.setVariables(variables);
					importOptionData.setxLo(importTask.getImportParameters().getxLo());
					importOptionData.setxHi(importTask.getImportParameters().getxHi());
					importOptionData.setyLo(importTask.getImportParameters().getyLo());
					importOptionData.setyHi(importTask.getImportParameters().getyHi());
					importOptionData.setzLo(importTask.getImportParameters().getzLo());
					importOptionData.setzHi(importTask.getImportParameters().getzHi());

				}

				ExecutionData executionData = new ExecutionData();

				if (importTask.getLastExecution() != null) {
					String timeBegin = null;
					try {
						Calendar calendarBegin = importTask.getLastExecution().getBegin();
						if (calendarBegin != null) {
							timeBegin = TimeUtils.toString(importTask.getLastExecution().getBegin());
						}
					} catch (Throwable e) {
						logger.error(e.getLocalizedMessage(), e);
					}

					String timeEnd = null;
					try {
						Calendar calendarEnd = importTask.getLastExecution().getEnd();
						if (calendarEnd != null) {
							timeEnd = TimeUtils.toString(importTask.getLastExecution().getEnd());
						}
					} catch (Throwable e) {
						logger.error(e.getLocalizedMessage(), e);
					}

					String timeLast = null;
					try {
						Calendar calendarLast = importTask.getLastExecution().getLastUpdate();
						if (calendarLast != null) {
							timeLast = TimeUtils.toString(importTask.getLastExecution().getLastUpdate());
						}
					} catch (Throwable e) {
						logger.error(e.getLocalizedMessage(), e);
					}

					executionData.setId(importTask.getLastExecution().getId());
					executionData.setBegin(timeBegin);
					executionData.setEnd(timeEnd);
					executionData.setLastUpdate(timeLast);
					executionData.setStatus(importTask.getLastExecution().getStatus().value());
					executionData.setProgress(importTask.getLastExecution().getProgress());

					ArrayList<ExecutionReportData> reports = new ArrayList<>();

					if (importTask.getLastExecution().getReports() != null
							&& !importTask.getLastExecution().getReports().isEmpty()) {

						for (ExecutionReport r : importTask.getLastExecution().getReports()) {
							ExecutionReportData report = new ExecutionReportData(r.getName(), r.getText(),
									r.getSnippet(), r.getSize());
							reports.add(report);
						}
					}
					executionData.setReports(reports);

				}

				taskData = new ImportTaskData(importTask.getId(), submissionInfoData, importOptionData, executionData);

			}

			return taskData;

		} catch (Throwable e) {
			logger.error("Error in CCmdCMEMSScheduleTask(): " + e.getLocalizedMessage(), e);
			throw new ServiceException("Error in CCmdCMEMSScheduleTask(): " + e.getLocalizedMessage(), e);

		}
	}

}
