package org.gcube.data.analysis.statisticalmanager.persistence;

import static org.gcube.data.analysis.statisticalmanager.persistence.PersistentHibernateManager.closeSession;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.collections.iterators.EntrySetMapIterator;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.analysis.statisticalmanager.operation.OperationStatus;
import org.gcube.data.analysis.statisticalmanager.operation.exception.SMFileManagerException;
import org.gcube.data.analysis.statisticalmanager.operation.importer.FileManager;
import org.gcube.data.analysis.statisticalmanager.persistence.data.Computation;
import org.gcube.data.analysis.statisticalmanager.persistence.data.FileOutput;
import org.gcube.data.analysis.statisticalmanager.persistence.data.ObjectOutput;
import org.gcube.data.analysis.statisticalmanager.persistence.data.Output;
import org.gcube.data.analysis.statisticalmanager.persistence.data.OutputType;
import org.gcube.data.analysis.statisticalmanager.persistence.data.TableMetadata;
import org.gcube.data.analysis.statisticalmanager.persistence.data.TabularOutput;
import org.gcube.data.analysis.statisticalmanager.persistence.data.User;
import org.gcube.data.analysis.statisticalmanager.persistence.exception.SMDataPersistenceException;
import org.gcube.data.analysis.statisticalmanager.persistence.exception.SMDataPersistenceNotExists;
import org.gcube.data.analysis.statisticalmanager.wsresources.StatisticalManagerFactoryResource;
import org.gcube.data.analysis.statisticalmanager.wsresources.StatisticalManagerServiceResource;
import org.gcube.dataanalysis.ecoengine.configuration.INFRASTRUCTURE;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.PrimitiveTypes;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.ComputationConfig;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.ComputationOutput;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.ComputationStatus;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.ComputationalAgentClass;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.ItemHistory;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.ItemHistoryList;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMComputation;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMHistoryRequest;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMTableMetadata;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMTableMetadataList;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMTypeParameter;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.StatisticalServiceType;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.StringValues;


import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.iq80.leveldb.table.UserComparator;

public class UserHistoryManager {

	GCUBELog logger = new GCUBELog(UserHistoryManager.class);
	
	private String userLogin;
	
	public UserHistoryManager(String userLogin) {	
		
		this.userLogin = userLogin;
	}
	
	public long addComputation(final ComputationConfig config)
	throws SMDataPersistenceException {
		
		Session session = PersistentHibernateManager.
		getSessionFactory().openSession();
		
		Transaction t = null;
		long computationId = 0;
		try {	
			t = session.beginTransaction();
			
			logger.debug("create Computation");
			Computation comp = new Computation(config.getComputation().getCategory().getValue(),
					config.getComputation().getAlgorithm(),config.getComputation().getDescription(),new Date());
			session.save(comp);
			
			User user = (User) session.load(User.class, userLogin);
			user.getComputations().add(comp);

			session.update(user);
			t.commit();
			computationId = comp.getComputationId();
		} catch(Exception e) {
			logger.error("Error in create computation ", e);
				if (t != null)
					t.rollback();
			throw new SMDataPersistenceException("Computation is not saved");
		} finally {
			closeSession(session);
		}
		return computationId;	
	}
	
	public ComputationOutput getComputationOutput(long computationId) throws SMDataPersistenceException {
		
		Session session = PersistentHibernateManager.
		getSessionFactory().openSession();

		try {
			Computation computation = (Computation) session.
			load(Computation.class, computationId);	
			Output output = computation.getOutput();
			ComputationOutput computationOutuput = convertOutputToComputationOutput(output);
			return computationOutuput;
		} catch (Exception e) {
			throw new SMDataPersistenceException("Computation output not found");
		} finally {
			closeSession(session);
		}
	}
	
	public void setStatusComputation(long computationId, OperationStatus status) 
	throws SMDataPersistenceException  {
		
		Session session = PersistentHibernateManager.
		getSessionFactory().openSession();
		
		Transaction t = null;
		try {
			t = session.beginTransaction();
		
			Computation comp = (Computation) session.load(Computation.class, computationId);
			comp.setStatus(status);
			comp.setEndDate(new Date());
			session.saveOrUpdate(comp);
		
			t.commit();
		} catch (Exception e) {
			
				if(t != null) 
					t.rollback();
			throw new SMDataPersistenceException("Statut is not set");
		} finally {
			closeSession(session);
		}
	}
	
	public void setComputationalInfrastructure(long computationId, INFRASTRUCTURE infrastructure) 
	throws SMDataPersistenceException  {
		
		Session session = PersistentHibernateManager.
		getSessionFactory().openSession();
		
		Transaction t = null;
		try {
			t = session.beginTransaction();
			Computation comp = (Computation) session.load(Computation.class, computationId);
			comp.setInfrastructure(infrastructure);
			session.saveOrUpdate(comp);
			t.commit();
		} catch (Exception e) {
				if(t != null) 
					t.rollback();
			throw new SMDataPersistenceException("Set conputational infrastructure failed");
		} finally {
			closeSession(session);
	
		}
	}
	
	public void setComputationObjectOutput(long computationId, String fileName, PrimitiveTypes objectType) throws SMDataPersistenceException {
		Session session = PersistentHibernateManager.
		getSessionFactory().openSession();
		
		Transaction t = null;
		try {
			t = session.beginTransaction();
		
			Computation comp = (Computation)session.load(Computation.class, computationId);
		
			Output output = new ObjectOutput(fileName, objectType);

			comp.setOutput(output);
			output.setComputation(comp);
		
			session.save(output);
			session.update(comp);

			t.commit();
		} catch (Exception e) {
				if (t != null)
					t.rollback();
				logger.error("Output not set", e);
			throw new SMDataPersistenceException("Outuput is not set ");
		} finally {
			closeSession(session);
		}
	}
	
	public void setComputationFileOutput(long computationId, String url, String fileName) throws SMDataPersistenceException {
		Session session = PersistentHibernateManager.
		getSessionFactory().openSession();
		
		Transaction t = null;
		try {
			t = session.beginTransaction();
		
			Computation comp = (Computation)session.load(Computation.class, computationId);
		
			Output output = new FileOutput(url, fileName);

			comp.setOutput(output);
			output.setComputation(comp);
		
			session.save(output);
			session.update(comp);

			t.commit();
		} catch (Exception e) {
				if (t != null)
					t.rollback();
			throw new SMDataPersistenceException("Outuput is not set ");
		} finally {
			closeSession(session);
		}
	}
	
	public void setComputationTabularOutput(long computationId, String tableId,
			String tableName) throws SMDataPersistenceException {
		
		Session session = PersistentHibernateManager.
		getSessionFactory().openSession();
		
		Transaction t = null;
		try {
			t = session.beginTransaction();
		
			Computation comp = (Computation)session.load(Computation.class, computationId);
		
			Output output = new TabularOutput(tableId, tableName);

			comp.setOutput(output);
			output.setComputation(comp);
		
			session.save(output);
			session.update(comp);

			t.commit();
		} catch (Exception e) {
				if (t != null)
					t.rollback();
			throw new SMDataPersistenceException("Outuput is not set ");
		} finally {
			closeSession(session);
		}
	}
	
	public ComputationOutput convertOutputToComputationOutput(Output output) throws SMFileManagerException {
				
		ComputationOutput out = new ComputationOutput();
		SMTypeParameter type = new SMTypeParameter();

		switch (output.getType()) {
		case TABULAR: {

			TabularOutput tabularOutput = (TabularOutput)output;
			out.setValue(tabularOutput.getTableName());

			type.setName(StatisticalServiceType.fromString(
					output.getType().toString()));
			String[] tamplates = {tabularOutput.getTableTemplate()};
			type.setValue(new StringValues(tamplates));
			out.setAlias(tabularOutput.getTableTemplate());

		}
		break;
		case FILE: {
			out.setValue(((FileOutput)output).getUrl());

			type.setName(StatisticalServiceType.fromString(
					output.getType().toString()));
			out.setAlias(((FileOutput)output).getFileName());
		} 
		break;
		case OBJECT: {
			ObjectOutput objectOutput = (ObjectOutput)output;
			out.setValue(objectOutput.getObjectFileName());

			type.setName(StatisticalServiceType.PRIMITIVE);
			String[] primitiveType = {objectOutput.getObjectType().toString()};
			type.setValue(new StringValues(primitiveType));			
		}
		break;
		default:
			break;
		}

		out.setType(type);
		return out;
		
	}
	
	private ItemHistory[] buildItemHistory(List<? extends Object> objects) throws SMFileManagerException {
		
		ItemHistory[] items = new ItemHistory[objects.size()];
		int i = 0;
		for (Object obj : objects) {
				
			Computation computation = (Computation)obj;
			SMComputation smComp = new SMComputation(computation.getAlgorithm(),
					ComputationalAgentClass.fromString(computation.getCategory()),computation.getDescription());
			
			items[i] = new ItemHistory();
			
			items[i].setComputationId(computation.getComputationId());
			Calendar startDate = Calendar.getInstance();
			
			startDate.setTime(computation.getStartDate());
			items[i].setStartDate(startDate);
			
			if (computation.getEndDate() != null) {
				Calendar endDate = Calendar.getInstance();
				endDate.setTime(computation.getEndDate());
				items[i].setEndDate(endDate);
			}
			
			items[i].setComputation(smComp);		
			items[i].setComputationStatus(ComputationStatus.
					fromString(computation.getStatus().toString()));
			
			if (computation.getInfrastructure() != null) {
				items[i].setInfrastructure(computation.getInfrastructure().toString());
			}
			
			Output output = computation.getOutput();
			if (output != null) {
				ComputationOutput compOut = convertOutputToComputationOutput(output);
				items[i].setOutput(compOut);
			}

			i++;
		}
		return items;
	}
	
	public ItemHistoryList getUserHistory() throws SMDataPersistenceException {
		
		Session session = PersistentHibernateManager.
		getSessionFactory().openSession();

		ItemHistory[] items = null;
		try {
		
			Query query = session.createQuery("select c from User u " +
					"join u.computations usercomputation, Computation c " +
					"where u.username = :name and c.computationId = usercomputation.computationId ");
			
			query.setParameter("name", userLogin);
			
			@SuppressWarnings("unchecked")
			List<Object> objects = query.list();
			items = buildItemHistory(objects);			
		
		} catch (Exception e) { 
			logger.error("-------- Exception " + e);
			throw new SMDataPersistenceException("User computations list not found " 
					+ e.getMessage());
		} finally {
			closeSession(session);
		}
		return new ItemHistoryList(items);	
	}
	
	
	public OperationStatus getComputationStatus(long computationId) throws SMDataPersistenceException {
		
		Session session = PersistentHibernateManager.
		getSessionFactory().openSession();

		try {
			Computation computation = (Computation) session.get(Computation.class, computationId);	
			if (computation == null)
				throw new SMDataPersistenceNotExists("computation with id" + computationId);
			return computation.getStatus();
		} catch (Exception e) {
			throw new SMDataPersistenceException(e.getMessage());
		} finally {
			closeSession(session);
		}
	}

	public OperationStatus deleteComputation(long computationId) throws SMDataPersistenceException {
		
		Session session = PersistentHibernateManager.
		getSessionFactory().openSession();

		OperationStatus status;
		Transaction t = null;
		try {
			t = session.beginTransaction();
		
			User user = (User) session.load(User.class, userLogin);
			Output output = (Output)session.get(Output.class, computationId);
			if(output != null)
				session.delete(output);

			Computation computation = (Computation) session.get(Computation.class, computationId);
			status = computation.getStatus();
			
			user.getComputations().remove(computation);
			
			session.delete(computation);
			t.commit();

		} catch (Exception e) {
			
			throw new SMDataPersistenceException(e.getMessage());
		} finally {
			closeSession(session);
		}
		
		return status;
	}

	public ItemHistoryList getUserHistory(
			SMHistoryRequest request) throws SMDataPersistenceException {
		
		Session session = PersistentHibernateManager.getSessionFactory().openSession();

		ItemHistory[] items = null;
		try {
			
			String hql =  "select c from User u " +
			"right join u.computations usercomputation, Computation c " +
			"where u.username = :name and c.computationId = usercomputation.computationId";
			
			if (request.getCategory() != null)
				hql += " and c.category = :category";
			if (request.getAlgorithm() != null)
				hql += " and c.algorithm = :algorithm";
			
			Query query = session.createQuery(hql);
			query.setParameter("name", userLogin);
			
			if (request.getCategory() != null)
				query.setParameter("category", request.getCategory());
			
			if (request.getAlgorithm() != null)
				query.setParameter("algorithm", request.getAlgorithm());
			
			@SuppressWarnings("unchecked")
			List<Object> objects = query.list();
			
			items = buildItemHistory(objects);
			
		} catch (Exception e) { 
			logger.error("-------- Exception " + e);
			throw new SMDataPersistenceException("User computations list not found " 
					+ e.getMessage());
		} finally {
			closeSession(session);
		}
		
		return new ItemHistoryList(items);
		
	}
	
}
