/**
 * 
 */
package org.gcube.data.trees.io;

import static javax.xml.stream.XMLStreamConstants.*;
import static org.gcube.data.trees.data.Nodes.*;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.gcube.data.trees.Constants;
import org.gcube.data.trees.data.Edge;
import org.gcube.data.trees.data.InnerNode;
import org.gcube.data.trees.data.Leaf;
import org.gcube.data.trees.data.Node;
import org.gcube.data.trees.data.Node.State;
import org.gcube.data.trees.data.Tree;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

/**
 *
 * Static facilities for binding trees to and from other data types.
 * 
 * @author Fabio Simeoni
 *
 */
public class Bindings {

	/** <null> value code.*/
	private static final String NULL="_null_";

	//factories
	private static final DocumentBuilderFactory domFactory = DocumentBuilderFactory.newInstance();
	private static final TransformerFactory tracFactory = TransformerFactory.newInstance();
	private static final XMLOutputFactory outputFactory=XMLOutputFactory.newInstance();
	private static final XMLInputFactory inputFactory=XMLInputFactory.newInstance();
	
	static {
		outputFactory.setProperty("javax.xml.stream.isRepairingNamespaces",new Boolean(true));
		inputFactory.setProperty("javax.xml.stream.isCoalescing",true);
		domFactory.setNamespaceAware(true);
	}
	
	//serialisation constants
	public static final String ROOT_NAME = "root";
	public static final String PREFIX = "t";
	public static final String URI="uri";
	public static final QName ROOT_QNAME = q(Constants.TREE_NS,ROOT_NAME);
	public static final String DEFAULT_NODE_NAME = "node";
	public static final QName NODE_QNAME = q(Constants.TREE_NS,DEFAULT_NODE_NAME);
	public static final String ID_ATTR = "id";
	public static final String STATUS_ATTR = "state";
	public static final String COLLID_ATTR = "collID";
	
	
	/**
	 * Binds a {@link Tree} to a {@link String}.
	 * @param t the tree
	 * @return the string
	 * @throws Exception if the binding fails
	 */
	public static String toText(Tree t) throws Exception {
		
		StringWriter writer = new StringWriter();
		toWriter(t,writer);
		return writer.toString();
	}
	
	/**
	 * Binds a {@link Tree} to a {@link Writer}.
	 * @param t the tree
	 * @param stream the writer
	 * @param writeDeclaration (optional) <code>true</code> if the binding is to include the XML declaration (default), 
	 * <code>false</code> otherwise
	 * @throws Exception if the binding fails
	 */
	public static void toWriter(Tree t, Writer stream, boolean ... writeDeclaration) throws Exception {
		
		XMLStreamWriter writer = outputFactory.createXMLStreamWriter(stream);
		bind(t, writer);
		writer.writeEndDocument();
		writer.flush();
		writer.close();
	}
	
	/**
	 * Binds a {@link Reader} to a {@link Tree}
	 * @param reader the reader
	 * @return the tree
	 * @throws Exception if the binding fails
	 */
	public static Tree fromReader(Reader reader) throws Exception {
		
		XMLStreamReader r = inputFactory.createXMLStreamReader(reader);
		r.nextTag(); //move to first element
		r.require(START_ELEMENT,Constants.TREE_NS,ROOT_NAME);
		return (Tree) bindReader(ROOT_QNAME,r);
	}
	
	/**
	 * Binds an {@link InputStream} to a {@link Tree}.
	 * @param stream stream
	 * @return the tree
	 * @throws Exception if the binding fails
	 */
	public static Tree fromStream(InputStream stream) throws Exception {
		return fromReader(new InputStreamReader(stream));
	}
	
	/**
	 * Binds a {@link Reader} to a {@link Node}.
	 * @param r the reader
	 * @return the node
	 * @throws Exception if the binding fails
	 */
	public static Node nodeFromReader(Reader r) throws Exception {
		
		XMLStreamReader reader = inputFactory.createXMLStreamReader(r);
		reader.nextTag(); //move to first element
		return bindReader(NODE_QNAME,reader);
	}
	
	/**
	 * Binds an {@link InputStream} to a {@link Node}.
	 * @param in the {@link InputStream}
	 * @return the {@link Node}
	 * @throws Exception if the binding fails
	 */
	public static Node nodeFromStream(InputStream in) throws Exception {
		return nodeFromReader(new InputStreamReader(in));
	}
	
	//dom
	
	/**
	 * Binds a {@link Tree} to an {@link Element}.
	 * @param tree the tree
	 * @return the element
	 * @throws Exception if the binding fails
	 */
	public static Element toElement(Tree tree) throws Exception {
		InputSource source = new InputSource(new StringReader(toText(tree)));
		return domFactory.newDocumentBuilder().parse(source).getDocumentElement();
	}
	

	/**
	 * Binds an {@link Element} to a {@link Tree}.
	 * @param element the element
	 * @return the tree
	 * @throws Exception if the binding fails
	 */
	public static Tree fromElement(Element element) throws Exception {
		
		if (element==null)
			return null;
		
		DOMSource source = new DOMSource(element);
		StringWriter writer = new StringWriter();
		StreamResult result = new StreamResult(writer);
		tracFactory.newTransformer().transform(source,result);
		return fromReader(new StringReader(writer.toString()));
	}

	/**
	 * Binds an {@link Element} to a {@link Node}.
	 * @param element the element
	 * @return the node
	 * @throws Exception if the binding fails
	 */
	public static Node nodeFromElement(Element element) throws Exception {
		
		if (element==null)
			return null;
		
		DOMSource source = new DOMSource(element);
		StringWriter writer = new StringWriter();
		StreamResult result = new StreamResult(writer);
		tracFactory.newTransformer().transform(source,result);
		return nodeFromReader(new StringReader(writer.toString()));
	}
	
	/**
	 * Binds a {@link Node} to an {@link Element}.
	 * @param node the node
	 * @param name (optional) a name for the element.
	 * @return the element
	 * @throws Exception if the binding fails
	 */
	public static Element nodeToElement(Node node, QName ... name) throws Exception {
		
		StringWriter w = new StringWriter();
		nodeToWriter(node,w,name);
		InputSource source = new InputSource(new StringReader(w.toString()));
		return domFactory.newDocumentBuilder().parse(source).getDocumentElement();

	}
	
	
	/**
	 * Binds a {@link Node} to a {@link Writer}.
	 * @param node the node
	 * @param stream the writer
	 * @param name (optional) an element name for the node
	 * @throws Exception if binding fails
	 */
	public static void nodeToWriter(Node node, Writer stream, QName ...name) throws Exception {		
		
		XMLStreamWriter writer = outputFactory.createXMLStreamWriter(stream);
		writer.setPrefix(PREFIX,Constants.TREE_NS);
		QName qname = name.length==0?NODE_QNAME:name[0];
		writer.writeStartDocument();
		
		if (node instanceof InnerNode)
			bindInner(qname,(InnerNode) node, writer);
		else
			bindLeaf(qname,(Leaf) node,writer);
		
		writer.writeEndDocument();
		
		writer.flush();
		writer.close();
		
	}
	
	//helper
	private static void bind(Tree t, XMLStreamWriter writer) throws Exception {
	
		writer.writeStartDocument();
		writer.writeStartElement(PREFIX,ROOT_NAME,Constants.TREE_NS);
		writer.writeNamespace(PREFIX,Constants.TREE_NS);
		
		bindNode(t,writer);
		
		if (t.sourceId()!=null)
			writer.writeAttribute(PREFIX,Constants.TREE_NS,COLLID_ATTR,t.sourceId());
		
		for (Edge e : t.edges())
			if (e.target() instanceof InnerNode)
				bindInner(e.label(),(InnerNode) e.target(), writer);
			else
				bindLeaf(e.label(),(Leaf) e.target(),writer);
		
		writer.writeEndElement();
	}

	private static void bindNode(Node node, XMLStreamWriter writer) throws Exception {
		
		for (Map.Entry<QName,String> attr : node.attributes().entrySet()) {
			String prefix = attr.getKey().getPrefix();
			String ns = attr.getKey().getNamespaceURI();
			String name = attr.getKey().getLocalPart();
			String value = attr.getValue()==null?NULL:attr.getValue();
			if (!prefix.isEmpty())
				writer.writeAttribute(prefix,ns,name,value);
			else if (!ns.isEmpty()) {
				writer.writeAttribute(ns,name,value);
			}
			else
				writer.writeAttribute(name,value);
		}
		
		if(node.id()!=null)
			writer.writeAttribute(PREFIX,Constants.TREE_NS,ID_ATTR,node.id());
		
		if(node.state()!=null)
			writer.writeAttribute(PREFIX,Constants.TREE_NS,STATUS_ATTR,node.state().name());
		
	}
	
	private static void bindInner(QName label, InnerNode node, XMLStreamWriter writer) throws Exception {

		if (!label.getPrefix().isEmpty())
			writer.writeStartElement(label.getPrefix(),label.getLocalPart(),label.getNamespaceURI());
		else if (!label.getNamespaceURI().isEmpty()) {
			writer.writeStartElement(label.getNamespaceURI(),label.getLocalPart());
		}
		else
			writer.writeStartElement(label.getLocalPart());
		
		bindNode(node, writer);
		
		for (Edge e : node.edges())
			if (e.target() instanceof InnerNode)
				bindInner(e.label(),(InnerNode) e.target(), writer);
			else
				bindLeaf(e.label(),(Leaf) e.target(),writer);
		
		writer.writeEndElement();
	}
	
	private static void bindLeaf(QName label, Leaf leaf, XMLStreamWriter writer) throws Exception {
		
		if (!label.getPrefix().isEmpty())
			writer.writeStartElement(label.getPrefix(),label.getLocalPart(),label.getNamespaceURI());
		else if (!label.getNamespaceURI().isEmpty()) {
			writer.writeStartElement(label.getNamespaceURI(),label.getLocalPart());
		}
		else
			writer.writeStartElement(label.getLocalPart());
		
		bindNode(leaf, writer);
		
		writer.writeCharacters(leaf.value()==null?NULL:leaf.value());
		
		writer.writeEndElement();
		
	}

	private static Node bindReader(QName label, XMLStreamReader reader) throws Exception {
		
		String collectionId=null;
		String id=null;
		State state=null;
		Map<QName,String> attributes = new HashMap<QName, String>();
		List<Edge> edges = new ArrayList<Edge>();
		String value=null;
		
		
		id = reader.getAttributeValue(Constants.TREE_NS,ID_ATTR);
		
		collectionId = reader.getAttributeValue(Constants.TREE_NS,COLLID_ATTR);
		
		String stringstate = reader.getAttributeValue(Constants.TREE_NS,STATUS_ATTR);
		if (stringstate!=null)
			state = State.valueOf(stringstate);
		
		for (int i=0;i<reader.getAttributeCount();i++) {
			String ns = reader.getAttributeNamespace(i);
			String name = reader.getAttributeLocalName(i);
			String prefix = reader.getAttributePrefix(i);
			
			if (ns!=null && ns.equals(Constants.TREE_NS))
				continue;
			
			QName attribute;
			if (prefix==null)
				attribute = new QName(ns,name,prefix);
			else
				attribute = new QName(ns,name);
			
			String attrValue = reader.getAttributeValue(i).equals(NULL)?null:reader.getAttributeValue(i);
			attributes.put(attribute,attrValue);
		}
		
		loop:while (reader.hasNext()) {
			
			boolean leaf = false;
			boolean inner = false;
			int next = reader.next();
			switch (next) {
				case START_ELEMENT:
					if (leaf)
						throw new IllegalArgumentException("invalid tree serialisation: found mixed content");
					inner=true;
					QName edgeLabel = reader.getName();
					edges.add(new Edge(edgeLabel,bindReader(edgeLabel,reader)));
					break;
				case CHARACTERS:
					if (inner)
						throw new IllegalArgumentException("invalid tree serialisation: found mixed content");
					leaf=true;
					value = reader.getText();
					break;
				case END_ELEMENT:
					break loop;
			}
		}
	
		if (label.equals(ROOT_QNAME)) {
			Tree doc = new Tree(id, state, attributes, edges.toArray(new Edge[0]));
			doc.setSourceId(collectionId);
			return doc;
		}
		
		if (value!=null) {
			if (value.equals(NULL))
				value=null;
			return new Leaf(id, state, value, attributes);
		}	
		return new InnerNode(id, state, attributes, edges.toArray(new Edge[0]));
	}

}
