package org.gcube.data.tm.services;

import static java.util.concurrent.TimeUnit.*;
import static org.gcube.data.tml.Constants.*;
import static org.gcube.data.tml.utils.Utils.*;
import gr.uoa.di.madgik.grs.buffer.IBuffer.Status;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPWriterProxy;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.field.Field;
import gr.uoa.di.madgik.grs.record.field.StringField;
import gr.uoa.di.madgik.grs.writer.GRS2WriterException;
import gr.uoa.di.madgik.grs.writer.RecordWriter;

import java.io.StringWriter;
import java.util.List;

import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.faults.GCUBERetryEquivalentFault;
import org.gcube.common.core.utils.handlers.GCUBEScheduledHandler.Mode;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.tm.activationrecord.ActivationRecord;
import org.gcube.data.tm.activationrecord.ActivationRecordBody;
import org.gcube.data.tm.context.ServiceContext;
import org.gcube.data.tm.context.TBinderContext;
import org.gcube.data.tm.publishers.ResilientScheduler;
import org.gcube.data.tm.state.TBinderResource;
import org.gcube.data.tm.stubs.BindOutcome;
import org.gcube.data.tm.stubs.BindOutcomeFailure;
import org.gcube.data.tm.stubs.BindParameters;
import org.gcube.data.tm.stubs.InvalidRequestFault;
import org.gcube.data.tm.stubs.SourceBinding;
import org.gcube.data.tm.stubs.SourceBindings;
import org.gcube.data.tml.Constants;
import org.gcube.data.tml.clients.BindingParameters;
import org.gcube.data.tml.exceptions.InvalidRequestException;
import org.globus.wsrf.encoding.ObjectSerializer;

/**
 * The implementation of the T-Binder service.
 * 
 * @author Fabio Simeoni
 *
 */
public class TBinderService {

	private static GCUBELog logger = new GCUBELog(TBinderService.class); 
	
	/**
	 * Returns {@link SourceBindings} from client parameters.
	 * @param parameters the parameters
	 * @return the bindings
	 * @throws InvalidRequestfault if the parameters are malformed
	 * @throws GCUBEFault if the request fails for any other error
	 */
	//public,axis-specific version
	public SourceBindings bind(BindParameters parameters) throws InvalidRequestFault, GCUBEFault {
		return bind(new BindingParameters(parameters));
	}
	
	/**
	 * Returns {@link SourceBindings} from client parameters.
	 * @param parameters the parameters
	 * @return the bindings
	 * @throws InvalidRequestFault if the parameters are malformed
	 * @throws GCUBEFault if the request fails for any other error
	 */
	//internal,future-proof version
	SourceBindings bind(final BindingParameters parameters) throws InvalidRequestFault, GCUBEFault {
		
		try {
		
			if (parameters==null)
				throw new InvalidRequestException("request carries null parameters");
				
			String pluginName = parameters.getPlugin();
			
			TBinderResource binder = TBinderContext.getContext().binder();
			
			if (parameters.isBroadcast() && binder.getPlugin(pluginName).isAnchored())
				throw new InvalidRequestException("cannot broadcast: plugin is anchored, its state cannot be replicated");
		
			//dispatch to factory
			List<SourceBinding> bindings = binder.bind(pluginName, parameters.getPayload());		
			
			//broadcast AR asynchronously and when readers/writers are staged (if there are actually any!)
			if (bindings.size()>0 && parameters.isBroadcast())
				buildAndPublishActivationRecord(binder, parameters);

			return new SourceBindings(bindings.toArray(new SourceBinding[0]));
		}
		catch(InvalidRequestException e) {
			throw newFault(new InvalidRequestFault(),e);
		}
		catch (Exception e) {
			throw newFault(new GCUBERetryEquivalentFault(),e);
	    }
			
	}	
	
	//helper
	private void buildAndPublishActivationRecord(TBinderResource binder,BindingParameters parameters) {
	
		try {
			
			//build AR
			String id = ServiceContext.getContext().getInstance().getID();
			ActivationRecordBody body = new ActivationRecordBody(id,parameters);
			String description = "An activation of the T-Binder Service";
			ActivationRecord record = ActivationRecord.newInstance(description,body);
			
			ResilientScheduler scheduler = new ResilientScheduler(1,Mode.LAZY);
			scheduler.setAttempts(Constants.MAX_ACTIVATIONRECORD_PUBLICATION_ATTEMPTS);
			scheduler.setDelay(10L);
			scheduler.setScopeManager(ServiceContext.getContext());
			scheduler.setSecurityManager(ServiceContext.getContext());
			record.publish(scheduler);
			
			//remember AR
			binder.addActivation(record);
		} 
		catch(Throwable t) {
			logger.error("could not publish activation record "+parameters,t);
		}
	}
	
	/**
	 * Returns the locator of a Result Set with the outcome of an invocation of {@link #bind(BindParameters)}.
	 * <p>
	 * The outcome is represented by a {@link BindOutcome} instance and it constitutes the only
	 * record in the Result Set. 
	 *
	 * @param parameters the input parameters to {@link #bind(BindParameters)}
	 * @return the locator of the Result Set with the outcome
	 * @throws GCUBEFault if the Result 
	 * 
	 * @see #bind(BindParameters)
	 */
	public String bindAsync(BindParameters parameters) throws GCUBEFault {
	
		
		try {
			
			final RecordWriter<GenericRecord> writer=
				new RecordWriter<GenericRecord>(new TCPWriterProxy(),UNTYPED_RECORD);
			
			AsynchronousBinder binder = new AsynchronousBinder(writer, parameters);
			
			ServiceContext.getContext().newServiceThread(binder).start();
			
			return writer.getLocator().toString();

		}		
		catch(GRS2WriterException e) {
			throw newFault(new GCUBERetryEquivalentFault(),e); //optimistically assume Result Set may work elsewhere
		}
		
	}
	
	//helper
	private class AsynchronousBinder implements Runnable {
		
		private final RecordWriter<GenericRecord> writer;
		private final BindParameters parameters;
		
		AsynchronousBinder(RecordWriter<GenericRecord> writer, BindParameters parameters) {
			this.writer=writer;
			this.parameters=parameters;
		}
		
		public void run() {
			
			BindOutcome response = new BindOutcome();
			
			try {
				response.setSuccess(bind(parameters));
			}
			catch(InvalidRequestFault f) {
				response.setFailure(new BindOutcomeFailure(f));
			}
			catch(Exception e) {
				response.setFailure(
						new BindOutcomeFailure(ServiceContext.getContext().getDefaultException(e).toFault())
				);
			}
			finally {
			
				try {
					if(writer.getStatus()==Status.Open) {
						StringWriter w = new StringWriter();
						ObjectSerializer.serialize(w,response,BindOutcome.getTypeDesc().getXmlType());
						GenericRecord record = new GenericRecord();
						record.setFields(new Field[]{new StringField(w.toString())});
						writer.put(record,5,SECONDS);
					}
				}
				catch (Exception e) {
					logger.error("could not write out asynchronous response ",e);
				}
				
				if(writer.getStatus()!=Status.Dispose) 
		        	try {
		        		writer.close();
					} 
					catch (GRS2WriterException e) {
						logger.warn("error closing the resultset", e);
					}
			}
				
		}
	}
		
	
}
