/**
 * 
 */
package org.gcube.data.tml.utils;

import static org.gcube.data.streams.dsl.Streams.*;

import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;

import org.gcube.data.streams.Stream;
import org.gcube.data.streams.dsl.publish.PublishRsWithClause;
import org.gcube.data.streams.generators.Generator;
import org.gcube.data.tm.stubs.Path;
import org.gcube.data.trees.data.Node;
import org.globus.wsrf.encoding.ObjectDeserializer;
import org.globus.wsrf.encoding.ObjectSerializer;
import org.xml.sax.InputSource;

/**
 * Result set conversion facilities.
 * 
 * @author Fabio Simeoni
 *
 */
public class TMStreams {

	
	/**
	 * Pipes an {@link Stream} of {@link Node}s through a logger.
	 * @param stream the stream
	 * @return the logged stream
	 */
	public static <N extends Node> Stream<N> log(Stream<N> stream) {
		TMStreamLogger<N> logger = new TMStreamLogger<N>();
		return monitor(pipe(stream).through(logger)).with(logger);
	}

	/**
	 * Converts a result set of {@link Path}s into a {@link Stream}.
	 * @param locator the result set locator
	 * @return the stream
	 */
	public static Stream<Path> pathsIn(URI locator) {
		return pipe(stringsIn(locator)).through(new PathParser());
	}
	
	/** 
	 * Publishes a {@link Stream} of {@link Path}s.
	 * @param stream the stream
	 * @return an ongoing publication sentence ready for further configuration
	 */
	public static PublishRsWithClause<Path> publishPathsIn(Stream<Path> stream) {
		return publish(stream).using(new PathSerialiser());
	}
	
	
	private static class PathParser implements Generator<String, Path> {

		public Path yield(String element)  {
			try {
				return (Path) ObjectDeserializer.deserialize(new InputSource(
					new StringReader(element)), Path.class);
			}
			catch(Exception e) {
				throw new RuntimeException(e);
			}
		}

	}

	
	private static class PathSerialiser implements Generator<Path,String> {
		
		public String yield(Path path) {
			
			try {
				
				StringWriter w = new StringWriter();
				
				ObjectSerializer.serialize(w, path, Path.getTypeDesc().getXmlType());
			
				return w.toString();
			}
			catch (Exception e) {//we take this serialisation failure to be unrecoverable
				throw new RuntimeException(e);
			}
		}
	};
	


}
