package org.gcube.data.tml.clients;

import java.net.URI;

import org.gcube.common.clients.DiscoveryException;
import org.gcube.data.streams.Stream;
import org.gcube.data.tm.stubs.UpdateFailure;
import org.gcube.data.tml.exceptions.InvalidTreeException;
import org.gcube.data.tml.exceptions.UnknownTreeException;
import org.gcube.data.tml.exceptions.UnsupportedOperationException;
import org.gcube.data.tml.exceptions.UnsupportedRequestException;
import org.gcube.data.trees.data.Tree;
import org.w3c.dom.Element;

/**
 * A high-level interface to T-Writer services.
 * <p>
 * T-Writer services give write-only access to given data sources under a tree-based data model.
 * <p>
 * Clients may target T-Writer services at known endpoints (<em>direct mode</em> ). Alternative, they may discover
 * T-Writer services bound to given sources ( <em>discovery mode</em>).
 * 
 * <p>
 * In discovery mode, clients attempt to interact with discovered services, until they succeed with one. They then cache
 * the the endpoint of this service and use it first for subsequent interactions.
 * 
 * @author Fabio Simeoni
 * @see TBinderClient
 * 
 */
public interface TWriterClient {

	/**
	 * Adds a {@link Tree} to the bound source.
	 * 
	 * @param tree the tree
	 * @return the tree after insertion.
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws InvalidTreeException if the tree cannot be added because it is invalid
	 * @throws Exception if the operation fails for an unexpected error
	 * 
	 */
	Tree add(Tree tree) throws DiscoveryException, UnsupportedOperationException, UnsupportedRequestException,
			InvalidTreeException, Exception;

	/**
	 * Adds a {@link Tree} to the bound source.
	 * 
	 * @param tree the {@link Element} representation of the tree
	 * @return the tree after insertion.
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws InvalidTreeException if the tree is invalid
	 * @throws Exception if the operation fails for an unexpected error
	 */
	Tree add(Element tree) throws DiscoveryException, UnsupportedOperationException, UnsupportedRequestException,
			InvalidTreeException, Exception;

	/**
	 * Adds zero or more {@link Tree}s to the bound source.
	 * 
	 * @param trees a {@link Stream} of {@link Tree}s
	 * @return a {@link Stream} of added {@link Tree}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 */
	Stream<Tree> add(Stream<Tree> trees) throws DiscoveryException, UnsupportedOperationException,
			UnsupportedRequestException, Exception;

	/**
	 * Adds zero or more {@link Tree}s to the bound source.
	 * 
	 * @param treeRs a locator to a Result Set of {@link Tree}s
	 * @return a {@link Stream} of added {@link Tree}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 */
	Stream<Tree> add(URI treeRs) throws DiscoveryException, UnsupportedOperationException, UnsupportedRequestException,
			Exception;

	/**
	 * Updates a tree in the bound source.
	 * 
	 * @param delta the delta tree that captures the update
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws InvalidTreeException if the delta tree is invalid
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws UnknownTreeException if the tree is unknown in the bound source (cannot be identified from the delta
	 *             tree)
	 * @throws Exception if the operation fails for an unexpected error
	 **/
	void update(Tree delta) throws DiscoveryException, UnsupportedOperationException, UnsupportedRequestException,
			InvalidTreeException, UnknownTreeException, Exception;

	/**
	 * Updates zero or more {@link Tree}s in the bound source.
	 * 
	 * @param deltas a {@link Stream} of {@link Tree}s to add
	 * @return a {@link Stream} of {@link UpdateFailure}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 * 
	 */
	Stream<UpdateFailure> update(Stream<Tree> deltas) throws DiscoveryException, UnsupportedOperationException,
			UnsupportedRequestException, Exception;
	
	/**
	 * Updates zero or more {@link Tree}s in the bound source.
	 * 
	 * @param deltaRs a locator to a Result Set of delta tree that capture the updates
	 * @return a {@link Stream} of {@link UpdateFailure}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 * 
	 */
	Stream<UpdateFailure> update(URI deltaRs) throws DiscoveryException, UnsupportedOperationException,
			UnsupportedRequestException, Exception;

}