package org.gcube.data.tml.clients;

import java.net.URI;

import org.gcube.common.clients.DiscoveryException;
import org.gcube.data.streams.Stream;
import org.gcube.data.tm.stubs.Path;
import org.gcube.data.tml.exceptions.UnknownPathException;
import org.gcube.data.tml.exceptions.UnknownTreeException;
import org.gcube.data.tml.exceptions.UnsupportedOperationException;
import org.gcube.data.tml.exceptions.UnsupportedRequestException;
import org.gcube.data.trees.data.Node;
import org.gcube.data.trees.data.Tree;
import org.gcube.data.trees.patterns.Pattern;
import org.w3c.dom.Element;

/**
 * A high-level interface to T-Reader services.
 * <p>
 * T-Reader services give read-only access to given data sources under a tree-based data model.
 * <p>
 * Clients may target T-Reader services at known endpoints (<em>direct mode</em> ). Alternative, they may discover
 * T-Reader services bound to given sources ( <em>discovery mode</em>).
 * 
 * <p>
 * In discovery mode, clients attempt to interact with discovered services, until they succeed with one. They then cache
 * the the endpoint of this service and use it first for subsequent interactions.
 * 
 * @author Fabio Simeoni
 * @see TBinderClient
 * 
 */
public interface TReaderClient {

	/**
	 * Returns a {@link Tree} in the bound source with a given identifier.
	 * 
	 * @param id the tree identifier
	 * @return the tree
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws UnknownTreeException if the identifier does not identify a tree in the bound source
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Tree get(String id) throws DiscoveryException, UnsupportedOperationException,
			UnsupportedRequestException, UnknownTreeException, Exception;

	/**
	 * Returns a {@link Tree} in the bound source with a given identifier, after pruning it with a {@link Pattern}.
	 * 
	 * @param id the tree identifier
	 * @param pattern the pattern
	 * @return the {@link Element} representation of the tree
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws UnknownTreeException if the identifier does not identify a tree in the bound source
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Tree get(final String id, Pattern pattern) throws DiscoveryException,
			UnsupportedOperationException, UnsupportedRequestException, UnknownTreeException, Exception;

	/**
	 * Returns a {@link Tree} in the bound source with a given identifier.
	 * 
	 * @param id the tree identifier
	 * @return the {@link Element} representation of the tree
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnknownTreeException if the identifier does not identify a tree in the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Element getAsElement(String id) throws DiscoveryException, UnsupportedOperationException,
			UnsupportedRequestException, UnknownTreeException, Exception;

	/**
	 * Returns a {@link Tree} in the bound source with a given identifier, after pruning it with a {@link Pattern}.
	 * 
	 * @param id the tree identifier
	 * @param pattern the pattern
	 * @return the {@link Element} representation of the tree
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws UnknownTreeException if the identifier does not identify a tree in the bound source
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Element getAsElement(final String id, Pattern pattern) throws DiscoveryException,
			UnsupportedOperationException, UnsupportedRequestException, UnknownTreeException, Exception;

	/**
	 * Returns a {@link Node} in the bound source from its path in a tree, i.e. the list of of one or more identifiers
	 * of the nodes that connect it to the root of the tree.
	 * 
	 * @param path the node identifiers
	 * @return the node
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws UnknownPathException if the identifiers do not identify a node in the bound source
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Node getNode(String... path) throws IllegalArgumentException, DiscoveryException,
			UnsupportedOperationException, UnsupportedRequestException, UnknownPathException, Exception;

	/**
	 * Returns a {@link Node} in the bound source from its path in a tree, Returns a {@link Node} from its in a tree,
	 * i.e. the list of identifiers of the nodes that connect it to the root of the tree.
	 * 
	 * @param ids the node identifiers.
	 * @return the {@link Element} representation of the node
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws UnknownPathException if the identifiers do not identify a node in the bound source
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Element getNodeAsElement(final String... ids) throws DiscoveryException,
			UnsupportedOperationException, UnsupportedRequestException, UnknownPathException, Exception;

	/**
	 * Returns {@link Tree}s in the bound source from their identifiers, after pruning them with a {@link Pattern}.
	 * 
	 * @param loc a locator to a Resultset of tree identifiers
	 * @param pattern the pattern
	 * @return a {@link Stream} of resolved {@link Tree}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 * 
	 */
	public abstract Stream<Tree> get(final URI loc, Pattern pattern) throws DiscoveryException,
			UnsupportedOperationException, UnsupportedRequestException, Exception;

	/**
	 * Returns {@link Tree}s in the bound source from their identifiers, after pruning them with a {@link Pattern}.
	 * 
	 * @param identifiers a {@link Stream} of tree identifiers
	 * @param pattern the pattern
	 * @return a {@link Stream} of resolved {@link Tree}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 * 
	 */
	public abstract Stream<Tree> get(final Stream<String> identifiers, Pattern pattern) throws DiscoveryException,
			UnsupportedOperationException, UnsupportedRequestException, Exception;

	/**
	 * Returns {@link Tree}s in the bound source with given identifiers.
	 * 
	 * @param loc a locator to a Result Set with tree identifiers
	 * @return a {@link Stream} of resolved {@link Tree}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Stream<Tree> get(URI loc) throws DiscoveryException, UnsupportedOperationException,
			UnsupportedRequestException, Exception;
	

	/**
	 * Returns {@link Tree}s in the bound source with given identifiers.
	 * 
	 * @param identifiers a {@link Stream} of tree identifiers
	 * @return a {@link Stream} of resolved {@link Tree}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Stream<Tree> get(Stream<String> identifiers) throws DiscoveryException, UnsupportedOperationException,
			UnsupportedRequestException, Exception;

	/**
	 * Returns all the {@link Tree}s in the bound source which match a given {@link Pattern}.
	 * 
	 * @param pattern the pattern
	 * @return a {@link Stream} of matching {@link Tree}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Stream<Tree> get(Pattern pattern) throws DiscoveryException, UnsupportedOperationException,
			UnsupportedRequestException, Exception;

	/**
	 * Returns all the {@link Tree}s in the bound source.
	 * 
	 * @return a {@link Stream} of matching {@link Tree}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Stream<Tree> get() throws DiscoveryException, UnsupportedOperationException, UnsupportedRequestException,
			Exception;

	/**
	 * Returns {@link Node}s from their paths in a tree, i.e. lists of identifiers of the nodes that connect them to the
	 * root of a tree.
	 * 
	 * @param paths a locator to a ResultSet of paths.
	 * @return a {@link Stream} of {@link Node}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Stream<Node> getNodes(final URI paths) throws DiscoveryException, UnsupportedOperationException,
			UnsupportedRequestException, Exception;
	
	/**
	 * Returns {@link Node}s from their paths in a tree, i.e. lists of identifiers of the nodes that connect them to the
	 * root of a tree.
	 * 
	 * @param paths a {@link Stream} of {@link Path}s
	 * @return a {@link Stream} of {@link Node}s
	 * @throws DiscoveryException if, in discovery mode, there is no service for the bound source
	 * @throws UnsupportedOperationException if the bound source does not support the operation
	 * @throws UnsupportedRequestException if the bound source does not support this particular request
	 * @throws Exception if the operation fails for an unexpected error
	 */
	public abstract Stream<Node> getNodes(final Stream<Path> paths) throws DiscoveryException, UnsupportedOperationException,
			UnsupportedRequestException, Exception;

}