/**
 * 
 */
package org.gcube.data.tmf.impl;

import static org.gcube.data.streams.dsl.Streams.*;

import org.gcube.data.streams.Stream;
import org.gcube.data.streams.exceptions.StreamSkipException;
import org.gcube.data.streams.generators.Generator;
import org.gcube.data.tmf.api.SourceWriter;
import org.gcube.data.tml.exceptions.InvalidTreeException;
import org.gcube.data.tml.exceptions.UnknownTreeException;
import org.gcube.data.tml.exceptions.UnsupportedOperationException;
import org.gcube.data.tml.outcomes.AddTreeOutcome;
import org.gcube.data.tml.outcomes.UpdateTreeFailure;
import org.gcube.data.trees.data.Tree;

/**
 * Partial implementation of {@link SourceWriter}.
 * <p>
 * Plugins that extend this class need only to implement
 * {@link SourceWriter#add(Tree)} and {@link SourceWriter#update(Tree)}, as the
 * implementation of the stream-based methods can be derived from these methods.
 * <p>
 * Note that the derived implementations are naive and highly inefficient if the
 * data source supports more direct implementations. In this case, plugins can
 * override derived implementations selectively.
 * 
 * @author Fabio Simeoni
 * 
 * @see SourceWriter
 * 
 */
public abstract class AbstractWriter implements SourceWriter {

	private static final long serialVersionUID = 1L;

	/**
	 * Default implementation of {@link SourceWriter#add(Stream)} based
	 * on repeated delegation to {@link SourceWriter#add(Tree)}.
	 * <p>
	 * It should be inherited only if the source does not support stream-based
	 * additions.
	 * 
	 * @see SourceWriter#add(Stream)
	 * */
	public Stream<AddTreeOutcome> add(Stream<Tree> stream)
			throws UnsupportedOperationException,Exception {

		Generator<Tree, AddTreeOutcome> addOne = new Generator<Tree, AddTreeOutcome>() {
			
			public AddTreeOutcome yield(Tree element) {
			
				AddTreeOutcome outcome;
				
				try {
					
					Tree tree = add(element);
					outcome = new AddTreeOutcome(tree);
				
				}
				catch (InvalidTreeException e) { //contingency
					outcome = new AddTreeOutcome(e);
				}
				catch(Exception e) { //unrecoverable
					throw new RuntimeException(e);
				}
				
				return outcome;
			}
		};
		
		return pipe(stream).through(addOne);
	}

	/**
	 * Default implementation of {@link SourceWriter#update(Stream)} based
	 * on repeated delegation to {@link SourceWriter#update(Tree)}.
	 * <p>
	 * It should be inherited only if the source does not support stream-based
	 * updates.
	 * 
	 * @see SourceWriter#update(Stream)
	 * */
	public Stream<UpdateTreeFailure> update(Stream<Tree> deltaStream)
			throws UnsupportedOperationException,Exception {

		Generator<Tree, UpdateTreeFailure> updateOne = new Generator<Tree, UpdateTreeFailure>() {
			public UpdateTreeFailure yield(Tree delta) {
				
				try {
				
					update(delta);
				
				} catch (UnknownTreeException e) {
					return new UpdateTreeFailure(delta.id(),e);
				}
				catch (InvalidTreeException e) {
					return new UpdateTreeFailure(delta.id(),e);
				}
				catch(Exception e) {
					throw new RuntimeException(e);
				}
				//update has been performed, skip this element
				throw new StreamSkipException();
			}
		};
		
		return pipe(deltaStream).through(updateOne);

	}

}
