/**
 * 
 */
package org.gcube.data.tmf.api;

import java.io.Serializable;

import org.gcube.data.streams.Stream;
import org.gcube.data.tml.exceptions.InvalidTreeException;
import org.gcube.data.tml.exceptions.UnknownTreeException;
import org.gcube.data.tml.exceptions.UnsupportedOperationException;
import org.gcube.data.tml.exceptions.UnsupportedRequestException;
import org.gcube.data.tml.outcomes.AddTreeOutcome;
import org.gcube.data.tml.outcomes.UpdateTreeFailure;
import org.gcube.data.trees.data.Tree;

/**
 * Writes, changes, or deletes trees in a data source.
 * <p>
 * The trees and the updates to the trees are expected to conform to the 'data
 * type' of the source.
 * 
 * @author Fabio Simeoni
 * @see Source#writer()
 * @see Tree
 * @see RemoteIterator
 */
public interface SourceWriter extends Serializable {

	/**
	 * Adds a tree to the source.
	 * 
	 * @param tree
	 *            the tree
	 * @return the input tree, possibly annotated with metadata added at the
	 *         time of insertion in the source.
	 * @throws InvalidTreeException
	 *             if the tree does not conform to the data type of the source
	 *             or if it is invalid for addition (e.g. some of its nodes have
	 *             already identifiers)
	 * @throws UnsupportedOperationException
	 *             if the data source does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the data source does not support this request
	 * @throws Exception
	 *             if the operation fails for any other reason
	 */
	Tree add(Tree tree) throws UnsupportedOperationException,
			UnsupportedRequestException, InvalidTreeException, Exception;

	/**
	 * Adds many trees at once to the source.
	 * 
	 * @param treeStream
	 *            the stream of trees
	 * @return a stream of outcomes, one per tree in the input stream and in the
	 *         same relative order.
	 * @throws UnsupportedOperationException
	 *             if the data source does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the data source does not support this request
	 * @throws Exception
	 *             if the operation fails as a whole for an unexpected error
	 */
	Stream<AddTreeOutcome> add(Stream<Tree> treeStream)
			throws UnsupportedOperationException, UnsupportedRequestException,
			Exception;

	/**
	 * Updates a tree in the source with the <em>delta tree</em> that captures
	 * the changes.
	 * 
	 * @param delta
	 *            the delta tree
	 * @throws UnknownTreeException
	 *             if the delta tree does not identify a tree in the source
	 * @throws InvalidTreeException
	 *             if the delta tree does now qualify for update
	 * @throws UnsupportedOperationException
	 *             if the data source does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the data source does not support this request
	 * @throws Exception
	 *             if the operation fails for any other error
	 */
	void update(Tree delta) throws UnsupportedOperationException,
			UnknownTreeException, UnsupportedRequestException,
			InvalidTreeException, Exception;

	/**
	 * Updates many trees at once in the source with the <em>delta trees</em>
	 * that capture their changes.
	 * 
	 * @param deltaStream
	 *            the delta trees
	 * @return the failures that may occur in the process
	 * @throws UnsupportedOperationException
	 *             if the data source does not support this operation
	 * @throws UnsupportedRequestException
	 *             if the data source does not support this request
	 * @throws Exception
	 *             if the operation fails as a whole for an expected error
	 */
	Stream<UpdateTreeFailure> update(Stream<Tree> deltaStream)
			throws UnsupportedOperationException, UnsupportedRequestException,
			Exception;
}
