package org.gcube.data.streams.handlers;

import org.gcube.data.streams.exceptions.StreamContingencyException;

/**
 * A partial implementation for {@link FaultHandler} that count failures and keep track of the last observed failure.
 * 
 * @author Fabio Simeoni
 * 
 */
public abstract class CountingHandler implements FaultHandler {

	private int count;
	private Exception lastFailure;

	@Override
	public final FaultResponse handle(RuntimeException failure) {
		FaultResponse outcome = handle(failure, lastFailure, count);
		count++;
		lastFailure = failure;
		return outcome;
	}

	/**
	 * Indicates whether iteration should continue or stop the iteration on the occurrence of an iteration failure.
	 * 
	 * @param failure the failure
	 * @param lastFailure the failure observed previously, or <code>null</code> if this is the first observed failure
	 * @param failureCount the number of failures counted so far, <code>0</code> if this is the first observed failure
	 * @return <code>true</code> if iteration should continue, <code>false</code> if it should stop
	 * @throws StreamContingencyException if no element can be yielded, because the input element violates
	 *             pre-conditions
	 * @throws RuntimeException if no element can be yielded from the input element, because of an unrecoverable
	 *             condition
	 */
	protected abstract FaultResponse handle(RuntimeException failure, Exception lastFailure, int failureCount)
			throws StreamContingencyException, RuntimeException;
}
