package org.gcube.data.access.storagehub.storage.backend.impl;

import java.io.BufferedInputStream;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;

import org.gcube.common.storagehub.model.exceptions.StorageIdNotFoundException;
import org.gcube.common.storagehub.model.items.nodes.Content;
import org.gcube.common.storagehub.model.items.nodes.PayloadBackend;
import org.gcube.common.storagehub.model.storages.MetaInfo;
import org.gcube.common.storagehub.model.storages.StorageBackend;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.amazonaws.ClientConfiguration;
import com.amazonaws.Protocol;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.client.builder.AwsClientBuilder.EndpointConfiguration;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.amazonaws.services.s3.transfer.TransferManagerBuilder;
import com.amazonaws.services.s3.transfer.Upload;
import com.amazonaws.util.IOUtils;


public class S3Backend extends StorageBackend{

	private static Logger log = LoggerFactory.getLogger(S3Backend.class);
	private final long minSizeThreshold = 10 * 1024 * 1024;
	private final long maxSizeThreshold = 100 * 1024 * 1024;
	
	// Connection tracking counters for leak detection
	private static final AtomicLong openedConnections = new AtomicLong(0);
	private static final AtomicLong closedConnections = new AtomicLong(0);
	
	Function<Void, String> keyGenerator; 
	String bucketName;
	AmazonS3 client;
	TransferManager transferManager;

	int maxConnections = -1;
	int connectionTimeout = -1;
	int socketTimeout = -1;
	int maxErrorRetry = -1;

	@Override
	protected void setPayloadConfiguration(PayloadBackend payloadConfiguration) {
		super.setPayloadConfiguration(payloadConfiguration);
	}

	public S3Backend(PayloadBackend payloadConfiguration, Function<Void, String> keyGenerator) {
		super(payloadConfiguration);
		this.keyGenerator = keyGenerator;
		Map<String, Object> parameters = payloadConfiguration.getParameters();
		this.bucketName = (String)parameters.get("bucketName");
		String accessKey = (String)parameters.get("key");
		String secret = (String)parameters.get("secret");
		String url = (String)parameters.get("url");
		boolean createBucket = Boolean.valueOf((String)parameters.get("createBucket"));

		try {
			AWSCredentials credentials = new BasicAWSCredentials(accessKey, secret);
			ClientConfiguration clientConfig = new ClientConfiguration();
			clientConfig.setProtocol(Protocol.HTTPS);

			if (parameters.containsKey("maxConnections")) {
				maxConnections = Integer.valueOf((String)parameters.get("maxConnections"));
				clientConfig.setMaxConnections(maxConnections);
			}
			if (parameters.containsKey("connectionTimeout")) {
				connectionTimeout = Integer.valueOf((String)parameters.get("connectionTimeout"));
				clientConfig.setConnectionTimeout(connectionTimeout);
			}
			if (parameters.containsKey("socketTimeout")) {
				socketTimeout = Integer.valueOf((String)parameters.get("socketTimeout"));
				clientConfig.setSocketTimeout(socketTimeout);
			}
			if (parameters.containsKey("maxErrorRetry")) {
				maxErrorRetry = Integer.valueOf((String)parameters.get("maxErrorRetry"));
				clientConfig.setMaxErrorRetry(maxErrorRetry);
			}
	
			log.debug("Initializing S3Backend - parameters are: bucketName = {}, url = {}, createBucket = {}, maxConnections = {}," +
				" connectionTimeout = {}, socketTimeout = {}, maxErrorRetry = {}",
			this.bucketName, url, createBucket, maxConnections,
			connectionTimeout, socketTimeout, maxErrorRetry);
		
			client = AmazonS3ClientBuilder.standard()
					.withEndpointConfiguration(new EndpointConfiguration(url,"us-east-1" ))
					.withCredentials(new AWSStaticCredentialsProvider(credentials))
					.enablePathStyleAccess()
					.withClientConfiguration(clientConfig).build(); 
			
			if (createBucket && !client.doesBucketExistV2(bucketName)) { 
				log.debug("creating {} bucket",this.bucketName);
				client.createBucket(bucketName);
				log.debug("bucket {} created",this.bucketName);
			} else log.debug("bucket not created");

			final int nThreads = 8;
			ExecutorService executorService = Executors.newFixedThreadPool(nThreads);
			transferManager = TransferManagerBuilder.standard()
                .withS3Client(client)
				.withMinimumUploadPartSize(maxSizeThreshold)
				.withMultipartUploadThreshold(maxSizeThreshold)
				.withExecutorFactory(() -> executorService)
				.build();

		} catch (Exception e) {
			log.error("error initializing s3",e);
			throw new RuntimeException("error initializing s3", e);
		} 
	}

	public boolean isAlive() {
		
		boolean toReturn = true;
		try {
			toReturn = client.doesBucketExistV2(bucketName);
		}catch (Exception e) {
			log.error("error checking aliveness",e);
			toReturn = false;
		}
		log.debug("the S3 backend is {} Alive",toReturn?"":"not");
		return toReturn;
	}
	
	@Override
	public MetaInfo onCopy(Content content, String newParentPath, String newName) {

		String sourceKey = content.getStorageId();
		String destinationKey = keyGenerator.apply(null);
		try {
			client.copyObject(bucketName, sourceKey, bucketName, destinationKey);
		} catch (Exception e) {
			throw new RuntimeException("error copying file on s3", e);
		} 
		return new MetaInfo(content.getSize(), destinationKey, null, getPayloadConfiguration());
	}

	@Override
	public MetaInfo onMove(Content content, String newParentPath) {
		//new contentPath can be set as remotePath to the storage backend ?
		return new MetaInfo(content.getSize(),content.getStorageId(), content.getRemotePath(), getPayloadConfiguration());
	}

	@Override
	public void delete(String storageId) {
		try {
			client.deleteObject(bucketName, storageId);
		} catch (Exception e) {
			throw new RuntimeException("error deleting file on s3", e);
		} 
	}
	

	@Override
	public MetaInfo upload(InputStream stream, String relativePath, String name, String user) {
		return this.upload(stream, relativePath, name, null, user);
	}

	@Override
	public MetaInfo upload(InputStream stream, String relativePath, String name, Long size, String user) {
			String storageId = keyGenerator.apply(null);
			return upload(stream, relativePath, name, storageId, size, user);
	}

	@Override
	public MetaInfo upload(InputStream stream, String relativePath, String name, String storageId, Long size, String user) {
		InputStream bufferedStream = null;
		try {
			
			long start = System.currentTimeMillis();
			
			ObjectMetadata objMeta = new ObjectMetadata();
			objMeta.addUserMetadata("user", user);
			objMeta.addUserMetadata("title", URLEncoder.encode(name, "UTF-8"));

			if (size != null && size > 0) {
				objMeta.setContentLength(size);
				log.info("content length set to {}", size);
			} else
				log.info("content length not set");

			log.info("uploading file {} with id {} in bucket {} ", name, storageId, bucketName);

			long sizeThreshold = Math.max(Math.min((Runtime.getRuntime().freeMemory() + Runtime.getRuntime().maxMemory() - Runtime.getRuntime().totalMemory()) / 5L, 
				maxSizeThreshold), minSizeThreshold);
			if (size != null && size > 0 && size < sizeThreshold)
				client.putObject(bucketName, storageId, stream, objMeta);
			else {
				bufferedStream = new BufferedInputStream(stream, (int)sizeThreshold + 1);
				Upload upload = transferManager.upload(bucketName, storageId, bufferedStream, objMeta);
	           	// Optionally, wait for the upload to finish before continuing.
	           	upload.waitForCompletion();
			}
			
			long fileSize;

			if (size != null && size > 0)
				fileSize = size;
			else 
				fileSize = client.getObjectMetadata(bucketName, storageId).getContentLength();

			// Drain the input stream to ensure all data is consumed
			try {
				IOUtils.drainInputStream(stream);
			} catch (Exception e) {
				log.warn("Error draining input stream for file {}", name, e);
			}
			
			log.info("uploading file {} in {} seconds",name, (System.currentTimeMillis()-start)/1000);
			return new MetaInfo(fileSize,storageId, null, getPayloadConfiguration());
		} catch (Exception e) {
			log.error("error uploading file {} on s3 to bucket {}", name, bucketName, e);
			throw new RuntimeException("error uploading file on s3", e);
		} finally {
			// Ensure streams are closed even if an error occurs
			if (bufferedStream != null) {
				try {
					bufferedStream.close();
				} catch (Exception e) {
					log.warn("Error closing buffered stream for file {}", name, e);
				}
			}
		}

	}
	
	@Override
	public InputStream download(String id) throws StorageIdNotFoundException{
		try {
			// CRITICAL FIX: Wrap S3Object to ensure proper connection cleanup
			// Without this, each download leaks one HTTP connection from the pool
			S3Object s3Object = client.getObject(bucketName, id);
			long opened = openedConnections.incrementAndGet();
			log.info("[CONN-TRACK] OPENED S3 connection for id: {} | Total opened: {} | Currently active: {}", 
				id, opened, opened - closedConnections.get());
			return new S3ObjectInputStreamWrapper(s3Object);
		}catch (Exception e) {
			log.error("error downloading file from s3 with id: {} from bucket: {}", id, bucketName, e);
			throw new RuntimeException("error downloading file from s3",e);
		} 
	}
	
	/**
	 * Called by S3ObjectInputStreamWrapper when a connection is closed.
	 */
	public static void incrementClosedConnections() {
		long closed = closedConnections.incrementAndGet();
		log.info("[CONN-TRACK] CLOSED S3 connection | Total closed: {} | Currently active: {}", 
			closed, openedConnections.get() - closed);
	}
	
	/**
	 * Returns the number of S3 connections opened since startup.
	 */
	public static long getOpenedConnections() {
		return openedConnections.get();
	}
	
	/**
	 * Returns the number of S3 connections closed since startup.
	 */
	public static long getClosedConnections() {
		return closedConnections.get();
	}
	
	/**
	 * Returns the number of currently active (opened but not closed) S3 connections.
	 */
	public static long getActiveConnections() {
		return openedConnections.get() - closedConnections.get();
	}

	@Override
	public InputStream download(Content content) throws StorageIdNotFoundException {
		return download(content.getStorageId());
	}

	@Override
	public Map<String, String> getFileMetadata(String id) {
		
		try {
			ObjectMetadata objMeta = client.getObjectMetadata(bucketName, id);
			Map<String, String> userMetadata = objMeta.getUserMetadata();
			HashMap<String, String> toReturn = new HashMap<>(userMetadata);			
			toReturn.put("size" , Long.toString(objMeta.getContentLength()));
			if (toReturn.containsKey("title")) {
				toReturn.put("title", URLEncoder.encode(toReturn.get("title"), "UTF-8"));
			}
			
			return toReturn;
		} catch (Exception e) {
			log.error("error getting metadata from s3");
			throw new RuntimeException("error downloading file from s3",e);
		} 
	}

	@Override
	public String getTotalSizeStored() {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public String getTotalItemsCount() {
		// TODO Auto-generated method stub
		return null;
	}

}
