package org.gcube.data.access.storagehub.storage.backend.impl;

import java.io.IOException;
import java.io.InputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;

/**
 * Wrapper for S3ObjectInputStream that ensures the underlying S3Object is properly closed
 * to prevent connection leaks. This is critical because S3Object holds an HTTP connection
 * that must be released back to the connection pool.
 */
public class S3ObjectInputStreamWrapper extends InputStream {

	private static final Logger log = LoggerFactory.getLogger(S3ObjectInputStreamWrapper.class);
	
	private final S3Object s3Object;
	private final S3ObjectInputStream objectContent;
	private boolean closed = false;
	
	public S3ObjectInputStreamWrapper(S3Object s3Object) {
		this.s3Object = s3Object;
		this.objectContent = s3Object.getObjectContent();
	}
	
	@Override
	public int read() throws IOException {
		return objectContent.read();
	}
	
	@Override
	public int read(byte[] b) throws IOException {
		return objectContent.read(b);
	}
	
	@Override
	public int read(byte[] b, int off, int len) throws IOException {
		return objectContent.read(b, off, len);
	}
	
	@Override
	public long skip(long n) throws IOException {
		return objectContent.skip(n);
	}
	
	@Override
	public int available() throws IOException {
		return objectContent.available();
	}
	
	@Override
	public void close() throws IOException {
		if (!closed) {
			closed = true;
			try {
				// First, try to abort the HTTP request to immediately release the connection
				if (objectContent != null) {
					try {
						objectContent.abort();
						log.debug("S3ObjectInputStream aborted successfully");
					} catch (Exception e) {
						log.warn("Error aborting S3ObjectInputStream, attempting normal close", e);
					}
				}
			} finally {
				// Always close the S3Object to ensure connection is released
				try {
					if (s3Object != null) {
						s3Object.close();
						log.debug("S3Object closed successfully, connection returned to pool");
					}
				} catch (Exception e) {
					log.error("Error closing S3Object - connection may leak!", e);
					throw e;
				}
			}
		}
	}
	
	@Override
	public synchronized void mark(int readlimit) {
		objectContent.mark(readlimit);
	}
	
	@Override
	public synchronized void reset() throws IOException {
		objectContent.reset();
	}
	
	@Override
	public boolean markSupported() {
		return objectContent.markSupported();
	}
}
