package org.gcube.data.access.storagehub.accounting;

import java.net.URI;
import java.net.URISyntaxException;
import org.gcube.accounting.datamodel.usagerecords.StorageUsageRecord;
import org.gcube.documentstore.exception.InvalidValueException;
import org.gcube.accounting.persistence.AccountingPersistence;
import org.gcube.accounting.persistence.AccountingPersistenceFactory;
import org.gcube.common.security.providers.SecretManagerProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.inject.Singleton;

@Singleton
public class ReportHandlerImpl implements ReportHandler {

	final Logger logger = LoggerFactory.getLogger(ReportHandlerImpl.class);
	private static final String DEFAULT_PROVIDER_URI = "data.d4science.org";
	private URI providerURI;

	public AccountingPersistence accountingPersistence;

	public ReportHandlerImpl() {
		this.accountingPersistence = AccountingPersistenceFactory.getPersistence();
		try {
			this.providerURI = new URI(DEFAULT_PROVIDER_URI);
		} catch (URISyntaxException e) {
			logger.error("Invalid provider URI: " + DEFAULT_PROVIDER_URI, e);
		}
	}

	@Override
	public StorageUsageRecord generateRecord(StorageUsageRecord sur, String consumerId,
		String operation, String operationResult, String owner, Long size) {
		logger.trace("Generating accounting record: consumerId: [" + consumerId
				+ "] operation: [" + operation + "] operationResult: [" + operationResult
				+ "] owner: [" + owner + "]" + " size: [" + size + "]");
		if (sur == null)
			sur = new StorageUsageRecord();

		try {
			sur.setScope(SecretManagerProvider.get().getContext());
			sur.setDataType(StorageUsageRecord.DataType.STORAGE);
			sur.setProviderURI(providerURI);

			if (consumerId != null)
				sur.setConsumerId(consumerId);
			if (owner != null)
				sur.setResourceOwner(owner);
			if (size != null)
				sur.setDataVolume(size);

			if (operation != null) {
				try {
					sur.setOperationType(convertOperation(operation));
				} catch (RuntimeException e) {
					logger.error("Error converting operation: " + operation, e);
				}
			}
			if (operationResult != null) {
				try {
					sur.setOperationResult(convertOperationResult(operationResult));
				} catch (RuntimeException e) {
					logger.error("Error converting operationResult: " + operationResult, e);
				}
			}
		} catch (InvalidValueException e) {
			logger.error("Error setting accounting record properties", e);
			return null;
		}

		logger.trace("Generated accounting record: " + sur);
		return sur;
	}

	private StorageUsageRecord.OperationResult convertOperationResult(String operationResult) {
		logger.debug("converting operationResult: " + operationResult);
		switch (operationResult) {
			case "SUCCESS":
				return StorageUsageRecord.OperationResult.SUCCESS;
			case "FAILED":
				return StorageUsageRecord.OperationResult.FAILED;
		}
		throw new RuntimeException("The operationResult " + operationResult + " is not converted or supported");
	}

	private StorageUsageRecord.OperationType convertOperation(String operation) {
		logger.debug("converting operation: " + operation);
		switch (operation) {
			case "UPLOAD":
				return StorageUsageRecord.OperationType.CREATE;
			case "DELETE":
				return StorageUsageRecord.OperationType.DELETE;
			case "MOVE":
				return StorageUsageRecord.OperationType.UPDATE;
			case "TRASH":
				return StorageUsageRecord.OperationType.UPDATE;
			case "RESTORE":
				return StorageUsageRecord.OperationType.UPDATE;
			case "COPY":
				return StorageUsageRecord.OperationType.CREATE;
			case "SOFT_COPY":
				return StorageUsageRecord.OperationType.CREATE;
			case "MOVE_DIR":
				return StorageUsageRecord.OperationType.UPDATE;
			case "COPY_DIR":
				return StorageUsageRecord.OperationType.CREATE;
			case "CREATE_DIR":
				return StorageUsageRecord.OperationType.CREATE;
			case "DOWNLOAD":
				return StorageUsageRecord.OperationType.READ;
		}
		throw new RuntimeException("The operation " + operation + " is not converted or supported");
	}

	@Override
	public void send(StorageUsageRecord sur) {
		logger.debug("report sending...");
		try {
			accountingPersistence.account(sur);
		} catch (InvalidValueException e) {
			logger.error("Error sending accounting record: " + sur, e);
		}
		logger.debug("report sent");
		logger.trace(sur.toString());
	}

	@Override
	public void setSuccessResult(StorageUsageRecord sur) {
		try {
			if (sur != null) {
				sur.setOperationResult(StorageUsageRecord.OperationResult.SUCCESS);
			}
		} catch (InvalidValueException e) {
			// just ignore, it cannot happen
		}
	}

	@Override
	public void setFailureResult(StorageUsageRecord sur) {
		try {
			if (sur != null) {
				sur.setOperationResult(StorageUsageRecord.OperationResult.FAILED);
			}
		} catch (InvalidValueException e) {
			// just ignore, it cannot happen
		}
	}

	@Override
	public void setSize(StorageUsageRecord sur, Long size) {
		try {
			if (sur != null && size != null) {
				sur.setDataVolume(size);
			}
		} catch (InvalidValueException e) {
			// just ignore, it cannot happen
		}
	}

	@Override
	public void setOwner(StorageUsageRecord sur, String owner) {
		try {
			if (sur != null && owner != null) {
				sur.setResourceOwner(owner);
			}
		} catch (InvalidValueException e) {
			// just ignore, it cannot happen
		}
	}

	@Override
	public void setOperationType(StorageUsageRecord sur, String operation) {
		try {
			if (sur != null && operation != null) {
				sur.setOperationType(convertOperation(operation));
			}
		} catch (Exception e) {
			// just ignore, let's keep the previous value
		}
	}

	@Override
	public void setConsumerId(StorageUsageRecord sur, String consumerId) {
		try {
			if (sur != null && consumerId != null) {
				sur.setConsumerId(consumerId);
			}
		} catch (InvalidValueException e) {
			// just ignore, it cannot happen
		}
	}

	public String getProviderUri() {
		return providerURI.toString();
	}
}
