package org.gcube.data.access.storagehub.accounting;

import org.gcube.accounting.datamodel.usagerecords.StorageUsageRecord;

/**
 * Interface for handling storage usage reporting and accounting operations.
 * Provides methods to configure, populate, and send storage usage records
 * for accounting purposes within the StorageHub system.
 */
public interface ReportHandler {

	/**
	 * Generates or updates a storage usage record with essential accounting information.
	 * This method creates a complete usage record suitable for accounting purposes by
	 * setting the core properties needed to track storage operations and resource consumption.
	 * 
	 * @param sur the storage usage record to populate (can be null, will create a new one if needed)
	 * @param consumerId the unique identifier of the user or service consuming the resource
	 * @param operation the type of storage operation performed (e.g., "UPLOAD", "DOWNLOAD", "DELETE", "MOVE")
	 * @param operationResult the outcome of the operation (e.g., "SUCCESS", "FAILURE")
	 * @param owner the username or identifier of the resource owner/creator
	 * @param size the amount of data involved in the operation, measured in kilobytes
	 * @return a fully populated StorageUsageRecord
	 */
	StorageUsageRecord generateRecord(StorageUsageRecord sur, String consumerId,
		String operation, String operationResult, String owner, Long size); 

	/**
	 * Sends the completed storage usage record to the accounting persistence layer
	 * for permanent storage and reporting purposes.
	 * 
	 * @param sur the fully populated storage usage record to send
	 * @throws RuntimeException if the accounting persistence is not available or fails
	 */
	void send(StorageUsageRecord sur);

	/**
	 * Sets the operation result to SUCCESS in the given storage usage record.
	 * 
	 * @param sur the storage usage record to update
	 */
	void setSuccessResult(StorageUsageRecord sur);

	/**
	 * Sets the operation result to FAILURE in the given storage usage record.
	 * 
	 * @param sur the storage usage record to update
	 */
	void setFailureResult(StorageUsageRecord sur);

	/**
	 * Sets the size in kilobytes for the given storage usage record.
	 * 
	 * @param sur the storage usage record to update
	 * @param size the size in kilobytes to set
	 */
	void setSize(StorageUsageRecord sur, Long size);

	/**
	 * Sets the owner for the resource associated to the given storage usage record.
	 * 
	 * @param sur the storage usage record to update
	 * @param owner the owner to set
	 */
	void setOwner(StorageUsageRecord sur, String owner);

	/**
	 * Sets the operation type for the given storage usage record based on the provided operation string.
	 * 
	 * @param sur the storage usage record to update
	 * @param operation the operation string to convert and set as operation type
	 */
	void setOperationType(StorageUsageRecord sur, String operation);

	/**
	 * Sets the consumer ID for the given storage usage record.
	 * 
	 * @param sur the storage usage record to update
	 * @param consumerId the consumer ID to set
	 */
	void setConsumerId(StorageUsageRecord sur, String consumerId);
}
