package org.gcube.data.access.storagehub.services;

import static org.gcube.data.access.storagehub.Roles.INFRASTRUCTURE_MANAGER_ROLE;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.util.Iterator;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.commons.compress.archivers.ArchiveException;
import org.gcube.common.authorization.control.annotations.AuthorizationControl;
import org.gcube.common.gxrest.response.outbound.GXOutboundErrorResponse;
import org.gcube.common.storagehub.model.exceptions.BackendGenericError;
import org.gcube.common.storagehub.model.exceptions.StorageHubException;
import org.gcube.common.storagehub.model.items.GCubeItem;
import org.gcube.common.storagehub.model.plugins.PluginParameters;
import org.gcube.data.access.storagehub.Constants;
import org.gcube.data.access.storagehub.StorageHubApplicationManager;
import org.gcube.data.access.storagehub.handlers.items.ItemHandler;
import org.gcube.data.access.storagehub.handlers.items.builders.ArchiveStructureCreationParameter;
import org.gcube.data.access.storagehub.handlers.items.builders.FileCreationParameters;
import org.gcube.data.access.storagehub.handlers.items.builders.FolderCreationParameters;
import org.gcube.data.access.storagehub.handlers.items.builders.GCubeItemCreationParameters;
import org.gcube.data.access.storagehub.handlers.items.builders.ItemsParameterBuilder;
import org.gcube.data.access.storagehub.handlers.items.builders.URLCreationParameters;
import org.gcube.data.access.storagehub.repository.StoragehubRepository;
import org.gcube.smartgears.annotations.ManagedBy;
import org.gcube.smartgears.utils.InnerMethodName;
import org.glassfish.jersey.media.multipart.FormDataContentDisposition;
import org.glassfish.jersey.media.multipart.FormDataParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.webcohesion.enunciate.metadata.rs.RequestHeader;
import com.webcohesion.enunciate.metadata.rs.RequestHeaders;

import jakarta.inject.Inject;
import jakarta.servlet.ServletContext;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.FormParam;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;


@Path("items")
@ManagedBy(StorageHubApplicationManager.class)
@RequestHeaders({
	  @RequestHeader( name = "Authorization", description = "Bearer token, see https://dev.d4science.org/how-to-access-resources"),
	})
public class ItemsCreator extends Impersonable{

	private static final Logger log = LoggerFactory.getLogger(ItemsCreator.class);

	@Context
	ServletContext context;

	private final StoragehubRepository repository = StoragehubRepository.repository;
	
	@Inject
	ItemHandler itemHandler;
	
	@POST
	@Consumes(MediaType.APPLICATION_FORM_URLENCODED)
	@Path("/{id}/create/FOLDER")
	public Response createFolder(@PathParam("id") String id, @FormParam("name") String name, 
			@FormParam("description") String description, @FormParam("hidden") boolean hidden) {
		InnerMethodName.set("createItem(FOLDER)");
		log.info("create folder item called");
		Session ses = null;
		String toReturn = null;
		try{
			ses = repository.getRepository().login(Constants.JCR_CREDENTIALS);
			ItemsParameterBuilder<FolderCreationParameters> builder = FolderCreationParameters.builder().name(name).description(description).hidden(hidden).on(id).with(ses).author(currentUser);
			toReturn =  itemHandler.create(builder.build());
		}catch(StorageHubException she ){
			log.error(she.getErrorMessage(), she);
			GXOutboundErrorResponse.throwException(she, Response.Status.fromStatusCode(she.getStatus()));
		}catch(RepositoryException re ){
			log.error("jcr error creating item", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error creating item", re));
		}catch(Throwable e ){
			log.error("unexpected error", e);
			GXOutboundErrorResponse.throwException(new BackendGenericError(e));
		}finally{
			if (ses!=null)
				ses.logout();

		}
		return Response.ok(toReturn).build();
	}

	@POST
	@AuthorizationControl(allowedRoles={INFRASTRUCTURE_MANAGER_ROLE})
	@Consumes(MediaType.APPLICATION_FORM_URLENCODED)
	@Path("/{id}/create/EXTERNALFOLDER")
	public Response createExternalFolder(@PathParam("id") String id, @FormParam("name") String name, 
			@FormParam("description") String description, @FormParam("hidden") boolean hidden,
			@FormParam("pluginName") String pluginName,
			@Context HttpServletRequest request) {
		InnerMethodName.set("createItem(EXTERNALFOLDER)");
		log.info("create folder item called");
		Session ses = null;
		String toReturn = null;
		try{
			Iterator<String> paramIt = request.getParameterNames().asIterator();
			Iterable<String> iterable = () -> paramIt;
			Stream<String> targetStream = StreamSupport.stream(iterable.spliterator(), false);
			
			PluginParameters pluginParams = new PluginParameters();
			
			targetStream.filter(v -> v.startsWith("plugin."))
					.forEach(v -> pluginParams.add(v.replace("plugin.", ""), request.getParameter(v)));
			
			log.debug("parameters for external folder with plugin {} are {}",pluginName, pluginParams.toString());
			ses = repository.getRepository().login(Constants.JCR_CREDENTIALS);
			ItemsParameterBuilder<FolderCreationParameters> builder = FolderCreationParameters.builder().name(name)
					.description(description).onRepository(pluginName).withParameters(pluginParams.getParameters()).hidden(hidden).on(id).with(ses).author(currentUser);
			toReturn =  itemHandler.create(builder.build());
		}catch(StorageHubException she ){
			log.error(she.getErrorMessage(), she);
			GXOutboundErrorResponse.throwException(she, Response.Status.fromStatusCode(she.getStatus()));
		}catch(RepositoryException re ){
			log.error("jcr error creating item", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error creating item", re));
		}catch(Throwable e ){
			log.error("unexpected error", e);
			GXOutboundErrorResponse.throwException(new BackendGenericError(e));
		}finally{
			if (ses!=null)
				ses.logout();

		}
		return Response.ok(toReturn).build();
	}

	@POST
	@Consumes(MediaType.APPLICATION_FORM_URLENCODED)
	@Path("/{id}/create/URL")
	public Response createURL(@PathParam("id") String id, @FormParam("name") String name, @FormParam("description") String description, @FormParam("value") URL value) {
		InnerMethodName.set("createItem(URL)");
		log.info("create url called");
		Session ses = null;
		String toReturn = null;
		try{
			ses = repository.getRepository().login(Constants.JCR_CREDENTIALS);

			ItemsParameterBuilder<URLCreationParameters> builder = URLCreationParameters.builder().name(name).description(description).url(value).on(id).with(ses).author(currentUser);

			toReturn =  itemHandler.create(builder.build());
		}catch(StorageHubException she ){
			log.error(she.getErrorMessage(), she);
			GXOutboundErrorResponse.throwException(she, Response.Status.fromStatusCode(she.getStatus()));
		}catch(RepositoryException re ){
			log.error("jcr error creating item", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error creating item", re));
		}catch(Throwable e ){
			log.error("unexpected error", e);
			GXOutboundErrorResponse.throwException(new BackendGenericError(e));
		}finally{
			if (ses!=null)
				ses.logout();

		}
		return Response.ok(toReturn).build();
	}


	@POST
	@Consumes(MediaType.APPLICATION_JSON)
	@Path("/{id}/create/GCUBEITEM")
	public String createGcubeItem(@PathParam("id") String id, GCubeItem item) {
		InnerMethodName.set("createItem(GCUBEITEM)");
		log.info("create Gcube item called");
		Session ses = null;
		String toReturn = null;

		try{
			ses = repository.getRepository().login(Constants.JCR_CREDENTIALS);
			ItemsParameterBuilder<GCubeItemCreationParameters> builder = GCubeItemCreationParameters.builder().item(item).on(id).with(ses).author(currentUser);

			toReturn =  itemHandler.create(builder.build());
		}catch(StorageHubException she ){
			log.error(she.getErrorMessage(), she);
			GXOutboundErrorResponse.throwException(she, Response.Status.fromStatusCode(she.getStatus()));
		}catch(RepositoryException re ){
			log.error("jcr error creating item", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error creating item", re));
		}catch(Throwable e ){
			log.error("unexpected error", e);
			GXOutboundErrorResponse.throwException(new BackendGenericError(e));
		}finally{
			if (ses!=null)
				ses.logout();
		}
		return toReturn;
	}


	@POST
	@Consumes(MediaType.APPLICATION_FORM_URLENCODED)
	@Path("/{id}/create/FILE")
	public String createFileItemFromUrl(@PathParam("id") String id, @FormParam("name") String name,
			@FormParam("description") String description,
			@FormParam("url") String url){
		InnerMethodName.set("createItem(FILEFromUrl)");

		Session ses = null;
		String toReturn = null;
		try{

			log.debug("UPLOAD: call started");
			ses = repository.getRepository().login(Constants.JCR_CREDENTIALS);
			
			URLConnection connectionURL = new URI(url).toURL().openConnection();
						
			long fileLength = connectionURL.getContentLengthLong();
						
			try(InputStream stream = connectionURL.getInputStream()){
				ItemsParameterBuilder<FileCreationParameters> builder = FileCreationParameters.builder().name(name).fileDetails(FormDataContentDisposition.name(name).size(fileLength).build())
						.description(description).stream(stream)
						.on(id).with(ses).author(currentUser);
				toReturn =  itemHandler.create(builder.build());
			}
			log.debug("UPLOAD: call finished");
		}catch(RepositoryException re ){
			log.error("jcr error creating file item", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error creating file item", re));
		}catch(StorageHubException she ){
			log.error(she.getErrorMessage(), she);
			GXOutboundErrorResponse.throwException(she, Response.Status.fromStatusCode(she.getStatus()));
		}catch(Throwable e ){
			log.error("unexpected error", e);
			GXOutboundErrorResponse.throwException(new BackendGenericError(e));
		}finally{
			if (ses!=null && ses.isLive()) {
				log.info("session closed");
				ses.logout();
			}
		}
		return toReturn;

	}

	@POST
	@Consumes(MediaType.MULTIPART_FORM_DATA)
	@Path("/{id}/create/FILE")
	public String createFileItem(@PathParam("id") String id, @FormDataParam("name") String name,
			@FormDataParam("description") String description,
			@FormDataParam("file") InputStream file,
			@FormDataParam("file") FormDataContentDisposition fileDetail){
		InnerMethodName.set("createItem(FILE)");

		Session ses = null;
		String toReturn = null;
		try(InputStream is = new BufferedInputStream(file)){

			long size = fileDetail.getSize();
			
			log.info("UPLOAD: call started with file size {}",size);
			ses = repository.getRepository().login(Constants.JCR_CREDENTIALS);
			ItemsParameterBuilder<FileCreationParameters> builder = FileCreationParameters.builder().name(name).description(description).stream(file).fileDetails(fileDetail)
					.on(id).with(ses).author(currentUser);
			log.debug("UPLOAD: item prepared");
			toReturn =  itemHandler.create(builder.build());
			log.debug("UPLOAD: call finished");
		}catch(RepositoryException re ){
			log.error("jcr error creating file item", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error creating file item", re));
		}catch(StorageHubException she ){
			log.error(she.getErrorMessage(), she);
			GXOutboundErrorResponse.throwException(she, Response.Status.fromStatusCode(she.getStatus()));
		}catch(Throwable e ){
			log.error("unexpected error", e);
			GXOutboundErrorResponse.throwException(new BackendGenericError(e));
		}finally{
			if (ses!=null && ses.isLive()) {
				log.info("session closed");
				ses.logout();
			}
		}
		return toReturn;

	}

		

	@POST
	@Consumes(MediaType.APPLICATION_FORM_URLENCODED)
	@Path("/{id}/create/ARCHIVE")
	public String uploadArchiveFromURL(@PathParam("id") String id, @FormParam("parentFolderName") String parentFolderName,
			@FormParam("url") String url){
		InnerMethodName.set("createItem(ARCHIVEFromURL)");

		Session ses = null;
		String toReturn = null;
		try{
						
			ses = repository.getRepository().login(Constants.JCR_CREDENTIALS);
			try(InputStream stream = new URI(url).toURL().openStream()){
				ItemsParameterBuilder<ArchiveStructureCreationParameter> builder = ArchiveStructureCreationParameter.builder().parentName(parentFolderName).stream(stream)
						.on(id).with(ses).author(currentUser);
				toReturn =  itemHandler.create(builder.build());
			}
			
		}catch(RepositoryException | ArchiveException | IOException re){
			log.error("jcr error extracting archive", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error extracting archive", re));
		}catch(StorageHubException she ){
			log.error(she.getErrorMessage(), she);
			GXOutboundErrorResponse.throwException(she, Response.Status.fromStatusCode(she.getStatus()));
		}catch(Throwable e ){
			log.error("unexpected error", e);
			GXOutboundErrorResponse.throwException(new BackendGenericError(e));
		} finally{
			if (ses!=null)
				ses.logout();

		}
		return toReturn;
	}

	@POST
	@Consumes(MediaType.MULTIPART_FORM_DATA)
	@Path("/{id}/create/ARCHIVE")
	public String uploadArchive(@PathParam("id") String id, @FormDataParam("parentFolderName") String parentFolderName,
			@FormDataParam("file") InputStream stream,
			@FormDataParam("file") FormDataContentDisposition fileDetail){
		InnerMethodName.set("createItem(ARCHIVE)");

		Session ses = null;
		String toReturn = null;
		try(InputStream is = new BufferedInputStream(stream)){
			ses = repository.getRepository().login(Constants.JCR_CREDENTIALS);

			ItemsParameterBuilder<ArchiveStructureCreationParameter> builder = ArchiveStructureCreationParameter.builder().parentName(parentFolderName).stream(is).fileDetails(fileDetail)
					.on(id).with(ses).author(currentUser);

			toReturn =  itemHandler.create(builder.build());

		}catch(RepositoryException | ArchiveException | IOException re){
			log.error("jcr error extracting archive", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error extracting archive", re));
		}catch(StorageHubException she ){
			log.error(she.getErrorMessage(), she);
			GXOutboundErrorResponse.throwException(she, Response.Status.fromStatusCode(she.getStatus()));
		}catch(Throwable e ){
			log.error("unexpected error", e);
			GXOutboundErrorResponse.throwException(new BackendGenericError(e));
		} finally{
			if (ses!=null)
				ses.logout();

		}
		return toReturn;
	}




}
