package org.gcube.data.access.storagehub.services;

import java.util.ArrayList;
import java.util.List;

import javax.jcr.RepositoryException;

import org.apache.jackrabbit.api.JackrabbitSession;
import org.gcube.common.authorization.control.annotations.AuthorizationControl;
import org.gcube.common.gxrest.response.outbound.GXOutboundErrorResponse;
import org.gcube.common.storagehub.model.exceptions.BackendGenericError;
import org.gcube.common.storagehub.model.exceptions.IdNotFoundException;
import org.gcube.common.storagehub.model.exceptions.StorageHubException;
import org.gcube.common.storagehub.model.service.UsersList;
import org.gcube.common.storagehub.model.types.SHUBUser;
import org.gcube.data.access.storagehub.Constants;
import org.gcube.data.access.storagehub.StorageHubApplicationManager;
import org.gcube.data.access.storagehub.repository.StoragehubRepository;
import org.gcube.data.access.storagehub.services.delegates.UserManagerDelegate;
import org.gcube.smartgears.annotations.ManagedBy;
import org.gcube.smartgears.utils.InnerMethodName;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.webcohesion.enunciate.metadata.rs.RequestHeader;
import com.webcohesion.enunciate.metadata.rs.RequestHeaders;

import jakarta.inject.Inject;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.FormParam;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

@Path("users")
@ManagedBy(StorageHubApplicationManager.class)
@RequestHeaders({
		@RequestHeader(name = "Authorization", description = "Bearer token, see https://dev.d4science.org/how-to-access-resources"), })
public class UserManager {

	private static final String INFRASTRUCTURE_MANAGER_ROLE = "Infrastructure-Manager";

	private static final Logger log = LoggerFactory.getLogger(UserManager.class);

	private final StoragehubRepository repository = StoragehubRepository.repository;

	@Inject
	UserManagerDelegate userHandler;

	@GET
	@Path("")
	@Produces(MediaType.APPLICATION_JSON)
	public UsersList getUsers() {
		InnerMethodName.set("getUsers");
		JackrabbitSession session = null;
		try {
			session = (JackrabbitSession) repository.getRepository().login(Constants.JCR_CREDENTIALS);
			return new UsersList(userHandler.getAllUsers(session));
		} catch (Throwable e) {
			log.error("jcr error getting users", e);
			GXOutboundErrorResponse.throwException(new BackendGenericError(e));
		} finally {
			if (session != null)
				session.logout();
		}
		return null;
	}

	@GET
	@Path("{user}")
	@Produces(MediaType.APPLICATION_JSON)
	public SHUBUser getUser(@PathParam("user") String user) {

		InnerMethodName.set("getUser");


		JackrabbitSession session = null;
		try {
			session = (JackrabbitSession) repository.getRepository().login(Constants.JCR_CREDENTIALS);
			return userHandler.getUser(session, user);
		} catch (StorageHubException se) {
			log.error("error getting user", se);
			GXOutboundErrorResponse.throwException(se);
		} catch (Exception e) {
			log.error("jcr error getting user", e);
			GXOutboundErrorResponse.throwException(new BackendGenericError(e));
		} finally {
			if (session != null)
				session.logout();
		}

		GXOutboundErrorResponse.throwException(new IdNotFoundException(user));

		return null;
	}

	@POST
	@Path("")
	@Consumes(MediaType.APPLICATION_FORM_URLENCODED)
	@AuthorizationControl(allowedRoles = { INFRASTRUCTURE_MANAGER_ROLE })
	public String createUser(@FormParam("user") String user, @FormParam("password") String password) {

		InnerMethodName.set("createUser");

		JackrabbitSession session = null;
		String userId = null;
		try {
			session = (JackrabbitSession) repository.getRepository().login(Constants.JCR_CREDENTIALS);

			userId = userHandler.createUser(session, user, password);

			session.save();
		} catch (StorageHubException she) {
			log.error(she.getErrorMessage(), she);
			GXOutboundErrorResponse.throwException(she, Response.Status.fromStatusCode(she.getStatus()));
		} catch (RepositoryException re) {
			log.error("jcr error creating item", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error creating item", re));
		} finally {
			if (session != null)
				session.logout();
		}

		return userId;
	}

	@PUT
	@Path("{user}")
	@Consumes(MediaType.APPLICATION_FORM_URLENCODED)
	@AuthorizationControl(allowedRoles = { INFRASTRUCTURE_MANAGER_ROLE })
	public String updateHomeUserToLatestVersion(@PathParam("user") String user) {

		InnerMethodName.set("updateHomeUserToLatestVersion");

		JackrabbitSession session = null;
		String userId = null;
		try {
			session = (JackrabbitSession) repository.getRepository().login(Constants.JCR_CREDENTIALS);

			userId = userHandler.updateHomeUserToLatestVersion(session, userId);
			
			session.save();
		} catch (StorageHubException she) {
			log.error(she.getErrorMessage(), she);
			GXOutboundErrorResponse.throwException(she, Response.Status.fromStatusCode(she.getStatus()));
		} catch (RepositoryException re) {
			log.error("jcr error creating item", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error creating item", re));
		} finally {
			if (session != null)
				session.logout();
		}

		return userId;
	}

	@DELETE
	@Path("{user}")
	@AuthorizationControl(allowedRoles = { INFRASTRUCTURE_MANAGER_ROLE })
	public String deleteUser(@PathParam("user") final String user) {

		InnerMethodName.set("deleteUser");

		JackrabbitSession session = null;
		try {

			session = (JackrabbitSession) repository.getRepository().login(Constants.JCR_CREDENTIALS);

			userHandler.deleteUser(session, user);
			
			session.save();
		} catch (StorageHubException she) {
			log.error(she.getErrorMessage(), she);
			GXOutboundErrorResponse.throwException(she, Response.Status.fromStatusCode(she.getStatus()));
		} catch (RepositoryException re) {
			log.error("jcr error removing item", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error removing item", re));
		} finally {
			if (session != null)
				session.logout();
		}

		return user;
	}

	@GET
	@Path("{user}/groups")
	@Produces(MediaType.APPLICATION_JSON)
	public List<String> getGroupsPerUser(@PathParam("user") final String user) {

		InnerMethodName.set("getGroupsPerUser");

		JackrabbitSession session = null;
		List<String> groups = new ArrayList<>();
		try {
			session = (JackrabbitSession) repository.getRepository().login(Constants.JCR_CREDENTIALS);

			userHandler.getGroupsPerUser(session, user);
		} catch (RepositoryException re) {
			log.error("jcr error creating item", re);
			GXOutboundErrorResponse.throwException(new BackendGenericError("jcr error creating item", re));
		} finally {
			if (session != null)
				session.logout();
		}
		return groups;
	}

	
}
