package org.gcube.data.access.storagehub;

import static org.gcube.common.storagehub.model.NodeConstants.ACCOUNTING_NAME;
import static org.gcube.common.storagehub.model.NodeConstants.CONTENT_NAME;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.version.Version;

import org.gcube.common.authorization.library.provider.AuthorizationProvider;
import org.gcube.common.storagehub.model.NodeConstants;
import org.gcube.common.storagehub.model.Paths;
import org.gcube.common.storagehub.model.exceptions.BackendGenericError;
import org.gcube.common.storagehub.model.items.AbstractFileItem;
import org.gcube.common.storagehub.model.items.FolderItem;
import org.gcube.common.storagehub.model.items.Item;
import org.gcube.common.storagehub.model.items.SharedFolder;
import org.gcube.common.storagehub.model.types.NodeProperty;
import org.gcube.contentmanager.storageclient.wrapper.AccessType;
import org.gcube.contentmanager.storageclient.wrapper.MemoryType;
import org.gcube.contentmanager.storageclient.wrapper.StorageClient;
import org.gcube.data.access.storagehub.accounting.AccountingHandler;
import org.gcube.data.access.storagehub.handlers.ItemHandler;
import org.gcube.data.access.storagehub.handlers.VersionHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Utils {

	public final static String SERVICE_NAME 				= "home-library";	
	public final static String SERVICE_CLASS 				= "org.gcube.portlets.user";
	private static final String FOLDERS_TYPE = "nthl:workspaceItem";
			
	private static final Logger logger = LoggerFactory.getLogger(Utils.class);

	public static String getSecurePassword(String user) throws Exception {
		String digest = null;
		try {
			MessageDigest md = MessageDigest.getInstance("MD5");
			byte[] hash = md.digest(user.getBytes("UTF-8"));

			//converting byte array to Hexadecimal String
			StringBuilder sb = new StringBuilder(2*hash.length);
			for(byte b : hash){
				sb.append(String.format("%02x", b&0xff));
			}
			digest = sb.toString();

		} catch (Exception e) {
			logger.error("error getting secure password",e);
		} 
		return digest;
	}
	
	public static long getItemCount(Node parent, boolean showHidden) throws RepositoryException{
		NodeIterator iterator = parent.getNodes();
		long count=0;
		while (iterator.hasNext()){
			Node current = iterator.nextNode();
			
			if (isToExclude(current, showHidden))
				continue;
			
			count++;
		}
		return count;
	}
	
	
	
	
	
	public static <T extends Item> List<T> getItemList(Node parent, List<String> excludes, Range range, boolean showHidden) throws RepositoryException, BackendGenericError{

		List<T> returnList = new ArrayList<T>();
		long start = System.currentTimeMillis();
		NodeIterator iterator = parent.getNodes();
		logger.trace("time to get iterator {}",(System.currentTimeMillis()-start));
		int count =0;
		logger.trace("selected range is {}", range);
		while (iterator.hasNext()){
			Node current = iterator.nextNode();
			
			if (isToExclude(current, showHidden))
				continue;
			
			if (range==null || (count>=range.getStart() && returnList.size()<range.getLimit())) {
				T item = ItemHandler.getItem(current, excludes);
				returnList.add(item);
			} 
			count++;
		}
		return returnList;
	}
	
	private static boolean isToExclude(Node node, boolean showHidden) throws RepositoryException{
		return ((node.getName().startsWith("rep:") || (node.getName().startsWith("hl:"))) || 
			(!showHidden && node.hasProperty(NodeProperty.HIDDEN.toString()) && node.getProperty(NodeProperty.HIDDEN.toString()).getBoolean()) ||
			(node.getPrimaryNodeType().getName().equals(FOLDERS_TYPE) && Constants.FOLDERS_TO_EXLUDE.contains(node.getName())));
	}
	
	public static org.gcube.common.storagehub.model.Path getHomePath(){
		return Paths.getPath(String.format("/Home/%s/Workspace",AuthorizationProvider.instance.get().getClient().getId()));
	}

	public static org.gcube.common.storagehub.model.Path getHomePath(String login){
		return Paths.getPath(String.format("/Home/%s/Workspace",login));
	}
	
	public static StorageClient getStorageClient(String login){
		return new StorageClient(SERVICE_CLASS, SERVICE_NAME, login, AccessType.SHARED, MemoryType.PERSISTENT);

	}
	
	public static Deque<Item> getAllNodesForZip(FolderItem directory,  Session session, AccountingHandler accountingHandler, List<String> excludes) throws RepositoryException, BackendGenericError{
		Deque<Item> queue = new LinkedList<Item>();
		Node currentNode = session.getNodeByIdentifier(directory.getId());
		queue.push(directory);
		Deque<Item> tempQueue = new LinkedList<Item>();
		logger.debug("adding directory {}",directory.getPath());
		for (Item item : Utils.getItemList(currentNode,Arrays.asList(NodeConstants.ACCOUNTING_NAME, NodeConstants.METADATA_NAME), null, false)){
			if (excludes.contains(item.getId())) continue;
			if (item instanceof FolderItem) 
				tempQueue.addAll(getAllNodesForZip((FolderItem) item, session, accountingHandler, excludes));
			else if (item instanceof AbstractFileItem){
				logger.debug("adding file {}",item.getPath());
				AbstractFileItem fileItem = (AbstractFileItem) item;
				accountingHandler.createReadObj(fileItem.getTitle(), session, session.getNodeByIdentifier(item.getId()), false);
				queue.addLast(item);
			}
		}
		queue.addAll(tempQueue);
		return queue;
	}


	public static void zipNode(ZipOutputStream zos, Deque<Item> queue, String login, org.gcube.common.storagehub.model.Path originalPath) throws Exception{
		logger.trace("originalPath is {}",originalPath.toPath());
		org.gcube.common.storagehub.model.Path actualPath = Paths.getPath("");
		while (!queue.isEmpty()) {
			Item item = queue.pop();
			if (item instanceof FolderItem) {
				actualPath = Paths.getPath(item.getPath());
				logger.trace("actualPath is {}",actualPath.toPath());
				String name = Paths.remove(actualPath, originalPath).toPath().replaceFirst("/", "");
				logger.trace("writing dir {}",name);
				zos.putNextEntry(new ZipEntry(name));
				zos.closeEntry();
			} else if (item instanceof AbstractFileItem){
				InputStream streamToWrite = Utils.getStorageClient(login).getClient().get().RFileAsInputStream(((AbstractFileItem)item).getContent().getStorageId());
				if (streamToWrite == null){
					logger.warn("discarding item {} ",item.getName());
					continue;
				}
				try(BufferedInputStream is = new BufferedInputStream(streamToWrite)){
					String name = Paths.remove(actualPath, originalPath).toPath()+item.getName().replaceFirst("/", "");
					logger.trace("writing file {}",name);
					zos.putNextEntry(new ZipEntry(name));
					copyStream(is, zos);

				}catch (Exception e) {
					logger.warn("error writing item {}", item.getName(),e);
				} finally{
					zos.closeEntry();
				}
			}
		}

	}

	private static void copyStream(InputStream in, OutputStream out) throws IOException {
		byte[] buffer = new byte[2048];
		int readcount = 0;
		while ((readcount=in.read(buffer))!=-1) {
			out.write(buffer, 0, readcount);
		}
	}
	
	public static boolean hasSharedChildren(FolderItem item, Session session) throws RepositoryException, BackendGenericError{
		Node currentNode = session.getNodeByIdentifier(item.getId());
		for (Item children : Utils.getItemList(currentNode,Arrays.asList(ACCOUNTING_NAME,CONTENT_NAME), null, false)){
			if (children instanceof FolderItem) 
				return (children instanceof SharedFolder) || hasSharedChildren((FolderItem)children, session);
		}
		return false;

	}
	
	
	public static void getAllContentIds(Session ses, Set<String> idsToDelete, Item itemToDelete, VersionHandler versionHandler) throws Exception{
		if (itemToDelete instanceof AbstractFileItem) {
			List<Version> versions = versionHandler.getContentVersionHistory(ses.getNodeByIdentifier(itemToDelete.getId()), ses);
			
			versions.forEach(v -> {
				try {
					String storageId =v.getProperty("hl:storageId").toString();
					idsToDelete.add(storageId);
					logger.info("retrieved StorageId {} for version {}", storageId, v.getName());
				} catch (Exception e) {
					logger.warn("error retreiving sotrageId",e);
				}
			});
			
			idsToDelete.add(((AbstractFileItem) itemToDelete).getContent().getStorageId());
		}else if (itemToDelete instanceof FolderItem) {
			List<Item> items = Utils.getItemList(ses.getNodeByIdentifier(itemToDelete.getId()), Arrays.asList(NodeConstants.ACCOUNTING_NAME, NodeConstants.METADATA_NAME, NodeConstants.OWNER_NAME) , null, true);
			for (Item item: items) 
				getAllContentIds(ses, idsToDelete, item, versionHandler);

		}

	}
	
}
