package org.gcube.data.access.sharelatex.connector.resources;

import static org.gcube.common.authorization.client.Constants.authorizationService;
import static org.gcube.resources.discovery.icclient.ICFactory.client;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;

import java.io.IOException;
import java.net.URI;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.ServletContext;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.NewCookie;
import javax.ws.rs.core.Response;

import lombok.extern.slf4j.Slf4j;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.bson.Document;
import org.bson.types.ObjectId;
import org.gcube.common.authorization.library.AuthorizationEntry;
import org.gcube.common.authorization.library.provider.AuthorizationProvider;
import org.gcube.common.authorization.library.provider.SecurityTokenProvider;
import org.gcube.common.resources.gcore.GCoreEndpoint;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.data.access.sharelatex.connector.User;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.impl.XQuery;
import org.mindrot.jbcrypt.BCrypt;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.mongodb.BasicDBObject;
import com.mongodb.MongoClient;
import com.mongodb.ServerAddress;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;

@Path("")
@Slf4j
public class Resource {

	private static final String LOGIN_URL_PARAM = "internalLoginUrl";
	private static final String HOST_NAME_PARAM = "hostName";
	private static final String MONGO_DATABASE_PARAM = "mongoDatabaseName";


	private static final String DOCS_COLLECTION_NAME = "docs";
	private static final String PROJECTS_COLLECTION_NAME = "projects";
	private static final String USER_COLLECTION_NAME = "users";
	private static final String NOTIFICATION_COLLECTION_NAME = "notifications";
	private static final String INVITATION_COLLECTION_NAME = "projectInvites";
	private static final String GCUBE_LOGIN_FIELD = "gcube_login";
	private static final String EMAIL_FIELD = "email";
	private static final String SIGNUP_DATE_FIELD = "signUpDate";
	private static final String LAST_LOGGEDIN__FIELD = "lastLoggedIn";

	private static final String LAST_NAME__FIELD = "last_name";
	private static final String FIRST_NAME_FIELD = "first_name";

	private static final String PWD_FIELD = "password";
	private static final String CSRF_TOKEN_FIELD = "_csrf";

	private static final String SHARELATEX_COOKIE_NAME = "sharelatex.sid";
	private static final String USER_ID_COOKIE = "_ga";

	private static String FIRST_PROJECT_NAME ="First Example";

	private static final String SEND_MESSAGE_METHOD ="/messages/write-message";
	private static final String GET_PROFILE_METHOD ="/users/get-profile";

	private static final String TOKEN_COOKIE_NAME = "gcube-key";

	private static final Map<String, ReentrantLock> justRegistered = Collections.synchronizedMap(new WeakHashMap<String, ReentrantLock>());

	@Context ServletContext context;

	@Path("connect")
	@GET
	public Response connect(@Context HttpServletRequest request, @Context HttpServletRequest response){
		ReentrantLock lock = null;
		try{
			String user =  AuthorizationProvider.instance.get().getClient().getId();
			log.info("request for connection from {}",user);
			synchronized (justRegistered) {
				if (justRegistered.containsKey(user))
					lock = justRegistered.get(user);
				else {
					lock = new ReentrantLock(true);
					justRegistered.put(user, lock);
				}
			}
			lock.lock();
			String userRealLogin = checkUser(user);
			if (userRealLogin==null){
				log.info("trying to register user {}",AuthorizationProvider.instance.get().getClient().getId());
				userRealLogin = registerUser();
			} else log.info("user {} already exists",AuthorizationProvider.instance.get().getClient().getId());
			return _connect(userRealLogin, request.isSecure());
		}catch(Throwable e){
			log.error("error trying to connect",e);
			return Response.serverError().build();
		} finally{
			lock.unlock();
		}

	}


	@Path("project/{projectId}/invite")
	@POST
	public Response invite(@PathParam("projectId") String projectId, String payload, @Context HttpServletRequest request, @Context HttpServletRequest response){

		log.info("inviting user to use project with id {} and payload {}",projectId,payload);
		try{
			ObjectMapper mapper = new ObjectMapper();
			Map<String, Object> map = new HashMap<String, Object>();
			// convert JSON string to Map
			map = mapper.readValue(payload, new TypeReference<Map<String, Object>>(){});
			String email = (String) map.get(EMAIL_FIELD);
			log.info("user email found is {}",email);

			String projectName = retrieveProjectNameById(projectId);

			String token = null;
			Cookie[] cookies = request.getCookies();
			for (Cookie cookie : cookies){
				if (cookie.getName().trim().equals(TOKEN_COOKIE_NAME)){
					token = cookie.getValue();					
					break;
				}
			}

			AuthorizationEntry entry = authorizationService().get(token);

			ScopeProvider.instance.set(entry.getContext());
			SecurityTokenProvider.instance.set(token);

			boolean messageSent = sendMessage(token, email, projectName);

			if (messageSent){
				String csrfToken = (String) map.get(CSRF_TOKEN_FIELD);
				String privileges = (String) map.get("privileges");
				log.debug("old payload was {} and new csrf is {} ",payload, csrfToken);
				String[] emails = email.split(",");
				
				callSharelatexInviteInternally(entry.getClientInfo().getId(), Arrays.asList(emails), csrfToken, privileges, projectId, projectName);
				return Response.ok().build();
			}else throw new Exception("error sending message via social networking service");			
		}catch(Exception e){
			log.error("error sending invites for a project",e);
			return Response.serverError().build();
		}finally{
			ScopeProvider.instance.reset();
			SecurityTokenProvider.instance.reset();
		}
	}

	private void callSharelatexInviteInternally(String senderLogin,List<String> emails, String csrfToken, String privileges, String projectId, String projectName) throws Exception{
		MongoClient client = connectToMongo();
		String mongoDB = context.getInitParameter(MONGO_DATABASE_PARAM);
		try{
			MongoDatabase db = client.getDatabase(mongoDB);
			MongoCollection<Document> usersCollection = db.getCollection(USER_COLLECTION_NAME);
			BasicDBObject senderIdQuery = new BasicDBObject();
			senderIdQuery.put(GCUBE_LOGIN_FIELD, senderLogin);
			Iterator<Document> userCursor = usersCollection.find(senderIdQuery).iterator();
			if (userCursor.hasNext()){
				Document userDoc = userCursor.next();
				String userId = ((ObjectId)userDoc.get("_id")).toString();
				Calendar now = Calendar.getInstance();
				SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ");
				String createdAt = sdf.format(now.getTime());
				now.add(Calendar.MONTH, 1);
				String expires = sdf.format(now.getTime());
				
				log.debug("found user id for login {} is {}",senderLogin, userId);
								
				String invitationJson = "{\"sendingUserId\" : ObjectId(\""+userId+"\"), \"projectId\" : ObjectId(\""+projectId+"\"), \"privileges\" : \""+privileges+"\"," 
						+"\"expires\" : ISODate(\""+expires+"\"), \"createdAt\" : ISODate(\""+createdAt+"\"), \"__v\" : 0 }";
				
				MongoCollection<Document> invitationColl = db.getCollection(INVITATION_COLLECTION_NAME);
				MongoCollection<Document> notificationColl = db.getCollection(NOTIFICATION_COLLECTION_NAME);
				List<Document> invitationDocs = new ArrayList<Document>(emails.size());
				List<Document> notificationDocs = new ArrayList<Document>(emails.size());
				for (String email: emails){
					ObjectId projectInviteId = new ObjectId();
					Document doc = Document.parse(invitationJson);
					doc.put(EMAIL_FIELD, email);
					doc.put("_id", projectInviteId);
					doc.put("token", csrfToken);
					invitationDocs.add(doc);
					
					BasicDBObject receiverQuery = new BasicDBObject();
					receiverQuery.put(EMAIL_FIELD, email);
					userCursor = usersCollection.find(receiverQuery).iterator();
					
					if (userCursor.hasNext()){
						userDoc = userCursor.next();
						ObjectId receiverId = (ObjectId)userDoc.get("_id");
						notificationDocs.add(createNotificationDoc(csrfToken, receiverId, senderLogin, expires, projectInviteId.toString(), projectId, projectName));
					}
				}
				invitationColl.insertMany(invitationDocs);
				notificationColl.insertMany(notificationDocs);
			}
		} finally{
			client.close();
		}
	}
	
	private Document createNotificationDoc(String token, ObjectId receiverId, String userLogin, String expires, String key, String projectId, String projectName ) throws Exception{
		String notificationJson = "{ \"key\" : \"project-invite-"+key+"\", \"messageOpts\" : "
				+ "{ \"userName\" : \""+userLogin+"\", \"projectName\" : \""+projectName+"\", \"projectId\" : ObjectId(\""+projectId+"\"), \"token\" : \""+token+"\" }," 
			+"\"templateKey\" : \"notification_project_invite\", \"expires\" : ISODate(\""+expires+"\") }";
		Document notificationDoc = Document.parse(notificationJson);
		notificationDoc.put("user_id", receiverId);
		return notificationDoc;
	}
	

	private String retrieveProjectNameById(String projectId) throws Exception{
		String mongoDB = context.getInitParameter(MONGO_DATABASE_PARAM);
		MongoClient client = connectToMongo();
		try{
			MongoDatabase db = client.getDatabase(mongoDB);
			MongoCollection<Document> projectCollection = db.getCollection(PROJECTS_COLLECTION_NAME);
			BasicDBObject whereQuery = new BasicDBObject();
			whereQuery.put("_id",new ObjectId(projectId));
			Iterator<Document> cursor = projectCollection.find(whereQuery).iterator();

			if (!cursor.hasNext())
				throw new Exception("project with id "+projectId+" not found");


			Document project = cursor.next();

			return (String)project.get("name");

		}finally{
			client.close();
		}
	}

	@Path("disconnect")
	@GET
	public Response disconnect(@Context HttpServletRequest request){
		try{
			log.info("request for disconnection ");			
			Cookie[] cookies = request.getCookies();
			NewCookie[] newCookies = new NewCookie[3];
			int index=0;
			if (cookies!=null){
				for (Cookie cookie: cookies){
					String name = cookie.getName();

					if (name.trim().equals(SHARELATEX_COOKIE_NAME) || name.trim().equals(USER_ID_COOKIE) || name.trim().equals(TOKEN_COOKIE_NAME) ){
						newCookies[index++] = new NewCookie(name, null, "/"  , cookie.getDomain(), cookie.getVersion(), 
								cookie.getComment(), 0, !cookie.isHttpOnly());
						log.debug("SHARELATEX cookie found");
					} 

				}

				return Response.ok(context.getClassLoader().getResourceAsStream("logout.html"))
						.cookie(newCookies).build();
			} else
				return Response.ok(context.getClassLoader().getResourceAsStream("logout.html")).build();

		}catch(Throwable e){
			log.error("error disconnecting",e);
			return Response.serverError().build();
		}

	}


	private Response _connect(String userRealLogin, boolean isSecureRequest){
		String hostName = context.getInitParameter(HOST_NAME_PARAM);
		String loginUrl = context.getInitParameter(LOGIN_URL_PARAM);
		String setCookies ="";
		HttpClient httpClient = new HttpClient();
		String csrfToken;
		try{
			csrfToken = retrieveCsrfToken(httpClient, loginUrl);
			setCookies = retrieveSharelatexCookie(httpClient, loginUrl, csrfToken, userRealLogin);
		}catch(Exception e){
			log.error("cannot retrieve csrf token",e);
			return Response.serverError().build();
		}


		String redirectUrl = "http"+(isSecureRequest?"s":"")+"://"+hostName+"/project";

		List<NewCookie> newCookies = elaborateCookies(setCookies);

		try{	
			return Response.seeOther(new URI(redirectUrl)).cookie(newCookies.toArray(new NewCookie[newCookies.size()])).build();
		}catch(Exception e){
			log.error("error trying to login",e);
			return Response.serverError().build();
		}
	}

	/**
	 * retrieve and set the sharelatex.sid cookie used by sharelatex to authenticate the user
	 * 
	 * @param setCookies
	 * @return
	 */
	private List<NewCookie> elaborateCookies(String setCookies) {
		List<NewCookie> newCookies = new ArrayList<NewCookie>();
		String[] cookies = setCookies.split(",",2);
		for (String cookie: cookies){
			String[] singleValues = cookie.split(";");
			String[] nameAndValue = singleValues[0].split("=");
			if (nameAndValue[0].trim().equals(SHARELATEX_COOKIE_NAME)){
				String pathValue = singleValues[1].split("=")[1].trim();
				newCookies.add(new NewCookie(nameAndValue[0].trim(), nameAndValue[1].trim(),pathValue , null, null, -1, false));
			} 
		}
		log.info("creating {} with token {}",TOKEN_COOKIE_NAME, SecurityTokenProvider.instance.get() );
		newCookies.add(new NewCookie(TOKEN_COOKIE_NAME, SecurityTokenProvider.instance.get(),"/", null, null, -1, true));
		return newCookies;

	}

	private String retrieveSharelatexCookie(HttpClient httpClient,
			String loginUrl, String csrfToken, String userRealLogin ) throws Exception{

		String commonPwd = getGeneratadePassword();

		PostMethod postMethod = new PostMethod(loginUrl);

		postMethod.setParameter(EMAIL_FIELD, userRealLogin);
		postMethod.setParameter(PWD_FIELD, commonPwd);
		postMethod.setParameter(CSRF_TOKEN_FIELD, csrfToken);

		log.info("try to login with user {} and password {}", userRealLogin, commonPwd);

		try {
			httpClient.executeMethod(postMethod);
		} catch (Exception e) {
			log.error("error trying to login",e);
			throw new Exception("error trying to login",e);
		} 

		log.info("returned status is {}",postMethod.getStatusText());
		if (postMethod.getStatusCode() != HttpStatus.SC_OK) {
			log.error("error on login response: error code is {} ",postMethod.getStatusCode());
			throw new Exception("error on login response: error code is "+postMethod.getStatusCode());
		}

		log.info("status text on login {} ",postMethod.getStatusText());
		try {
			log.info("response on login {} ",postMethod.getResponseBodyAsString());
		} catch (IOException e1) {
			log.warn("cannot log response body",e1);
		}
		return postMethod.getResponseHeader("Set-Cookie").getValue();
	}

	private String retrieveCsrfToken(HttpClient httpClient, String loginUrl) throws Exception{
		GetMethod getMethod = new GetMethod(loginUrl);
		httpClient.executeMethod(getMethod);
		if (getMethod.getStatusCode()==200){
			String response = getMethod.getResponseBodyAsString();
			Pattern pattern = Pattern.compile("input name=\""+CSRF_TOKEN_FIELD+"\" type=\"hidden\" value=\"([^\"]*)\"");
			Matcher matcher = pattern.matcher(response);
			if (matcher.find())
				return matcher.group(1);
			else throw new Exception("crsf not fount on the requeste login page");
		} else throw new Exception("the page "+loginUrl+" in not responding");


	}

	private String checkUser(String login){
		String mongoDB = context.getInitParameter(MONGO_DATABASE_PARAM);

		BasicDBObject whereQuery = new BasicDBObject();
		whereQuery.put(GCUBE_LOGIN_FIELD,login);
		log.info("checking for login {} in field {}",login, GCUBE_LOGIN_FIELD);
		MongoClient client = connectToMongo();
		try{
			MongoDatabase db = client.getDatabase(mongoDB);
			MongoCollection<Document> collection = db.getCollection(USER_COLLECTION_NAME);
			Iterator<Document> cursor = collection.find(whereQuery).iterator();
			if (!cursor.hasNext()){
				log.info("login {} not found",login);
				return null;
			} else{
				String emailField = (String)cursor.next().get(EMAIL_FIELD);
				log.info("login {} found with email field {}",login,emailField);
				return emailField;
			}
		}finally{
			client.close();
		}

	}

	private String registerUser() throws Exception{

		User user = retrieveUserInfo();

		log.trace("trying to register user {}",user.toString());

		String login =  AuthorizationProvider.instance.get().getClient().getId();
		String commonPwd = getGeneratadePassword();	
		String salt = BCrypt.gensalt(12);
		String hashedPwd = BCrypt.hashpw(commonPwd, salt);
		String mongoDB = context.getInitParameter(MONGO_DATABASE_PARAM);
		ObjectId userId = new ObjectId();

		MongoClient client = connectToMongo();
		try{
			MongoDatabase db = client.getDatabase(mongoDB);
			MongoCollection<Document> collection = db.getCollection(USER_COLLECTION_NAME);

			String jsonUser ="{ \"betaProgram\" : false, \"subscription\" : { \"hadFreeTrial\" : false }, "+ 
					"\"refered_user_count\" : 0, \"refered_users\" : [ ], \"referal_id\" : \"23bbb54a\", "+
					"\"features\" : { \"references\" : true, \"templates\" : true, \"compileGroup\" : \"standard\", "+
					"\"compileTimeout\" : 180, \"github\" : false, \"dropbox\" : true, \"versioning\" : true, \"collaborators\" : -1 }, "+
					"\"ace\" : { \"syntaxValidation\" : true, \"pdfViewer\" : \"pdfjs\", \"spellCheckLanguage\" : \"en\", \"autoComplete\" : true, "+
					"\"fontSize\" : 12, \"theme\" : \"textmate\", \"mode\" : \"none\" }, \"holdingAccount\" : false, \"loginCount\" : 0, "+
					"\"signUpDate\" : null, \"confirmed\" : false, \"isAdmin\" : false, "+
					"\"institution\" : \"\", \"role\" : \"\", \"last_name\" : \"\", \"first_name\" : \"\", \"email\" : \"\", \"__v\" : 1,"+ 
					"\"hashedPassword\" : null,"+ 
					"\"lastLoggedIn\" : null })";

			Document doc = Document.parse(jsonUser);

			doc.put("hashedPassword", hashedPwd);
			doc.put("_id",  userId);

			Calendar now = Calendar.getInstance();
			SimpleDateFormat sdfLogin = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ");

			String loginDate = sdfLogin.format(now.getTime());

			doc.put(SIGNUP_DATE_FIELD, "ISODate(\""+loginDate+"\")");
			doc.put(LAST_LOGGEDIN__FIELD, "ISODate(\""+loginDate+"\")");
			doc.put(EMAIL_FIELD, user.getEmail());
			doc.put(FIRST_NAME_FIELD, user.getFirstName() );
			doc.put(LAST_NAME__FIELD, user.getLastName());

			doc.put(GCUBE_LOGIN_FIELD, login);

			log.debug("inserting doc: {}",doc.toJson());

			collection.insertOne(doc);

			try{
				ObjectId projectId = new ObjectId();
				ObjectId docId = new ObjectId();

				log.debug("objectID is {}",userId);

				//insert project
				String projectJson = "{\"deletedDocs\" : [ ], \"description\" : \"\", \"deletedByExternalDataSource\" : false, "
						+ "\"spellCheckLanguage\" : \"en\", \"compiler\" : \"pdflatex\", \"publicAccesLevel\" : \"private\", "
						+ "\"rootFolder\" : [ { \"folders\" : [ ], \"fileRefs\" : [ ], "
						+ "\"docs\" : [ { \"_id\" : ObjectId(\""+docId+"\"), \"name\" : \"main.tex\" } ], "
						+ "\"name\" : \"rootFolder\" } ], \"readOnly_refs\" : [ ], \"collaberator_refs\" : [ ], \"active\" : true, "
						+ "\"lastUpdated\" : null , \"name\" : \""+FIRST_PROJECT_NAME+"\", \"__v\" : 0, \"rootDoc_id\" : ObjectId(\""+docId+"\"), "
						+ "\"lastOpened\" : null, \"archived\" : false }";

				MongoCollection<Document> projectCollection = db.getCollection(PROJECTS_COLLECTION_NAME);

				Document project = Document.parse(projectJson);
				project.put("_id", projectId);
				project.put("owner_ref", userId);
				project.put("lastUpdated", "ISODate("+loginDate+")");
				project.put("lastOpened", "ISODate("+loginDate+")");

				projectCollection.insertOne(project);

				MongoCollection<Document> docsCollection = db.getCollection(DOCS_COLLECTION_NAME);

				SimpleDateFormat sdfProject = new SimpleDateFormat("MMM yyyy");

				String docsJson = "{ \"lines\" : [ \"\\\\documentclass{article}\", "
						+ "\"\\\\usepackage[utf8]{inputenc}\", \"\", \"\\\\title{"+FIRST_PROJECT_NAME+"}\""
						+ ", \"\", \"\\\\author{"+user.getFirstName()+" "+user.getLastName()+"}\""
						+ ", \"\", \"\\\\date{"+sdfProject.format(now.getTime())+"}\""
						+ ", \"\", \"\\\\begin{document}\""
						+ ", \"\", \"\\\\maketitle\", \"\", \"\\\\section{Introduction}\", \"\", \"\\\\end{document}\", \"\" ], "
						+ "\"ranges\" : {  }, \"rev\" : 1 }";
				Document docs = Document.parse(docsJson);
				docs.put("project_id", projectId);
				docs.put("_id", docId);
				docsCollection.insertOne(docs);

			}catch(Exception e){
				log.warn("error creating first project",e);
			}

			return user.getEmail();
		}finally{
			client.close();
		}
	}

	private User retrieveUserInfo() throws Exception {
		//String socialServiceEnpoint = context.getInitParameter("socialServiceEnpoint");

		String socialServiceEnpoint = retrieveSocialServiceEnpoint();

		String name= "";
		String lastName = "unknown";
		String email = "";
		ObjectMapper mapper = new ObjectMapper();
		HttpClient httpClient = new HttpClient();
		GetMethod getProfile = new GetMethod(socialServiceEnpoint+GET_PROFILE_METHOD+"?gcube-token="+SecurityTokenProvider.instance.get());

		try {
			httpClient.executeMethod(getProfile);
			String profile = getProfile.getResponseBodyAsString();

			// convert JSON string to Map
			Map<String, Object> map = new HashMap<String, Object>();

			// convert JSON string to Map
			map = mapper.readValue(profile, new TypeReference<Map<String, Object>>(){});

			@SuppressWarnings("unchecked")
			Map<String, String> profileMap = (Map<String, String>)map.get("result");

			name = profileMap.get("first_name");

			if (profileMap.get("last_name")!=null && !profileMap.get("last_name").isEmpty())
				lastName = profileMap.get("last_name");

			email = profileMap.get("email");

			log.trace("found profile {}, {}, {}",name, lastName, email);

			return new User(name, lastName, email);
		} catch (Exception e) {
			log.warn("error getting profile from social service",e);
			throw e;
		} 
	}

	private boolean sendMessage(String token, String email, String projectName) throws Exception {

		String socialServiceEnpoint = retrieveSocialServiceEnpoint();

		PostMethod putMessage = new PostMethod(socialServiceEnpoint+SEND_MESSAGE_METHOD+"?gcube-token="+token);
		String jsonRequest = String.format("{\"subject\":\"%s\", \"body\":\"%s %s\", \"recipients\":[\"%s\"]}","sharleatex project invitation","invitation to project",projectName, email);

		putMessage.setRequestEntity(new StringRequestEntity(jsonRequest, "application/json" , "UTF-8"));

		log.debug("json request is {}", jsonRequest);

		HttpClient httpClient = new HttpClient();

		int returnedStatus = -1;
		try {
			returnedStatus = httpClient.executeMethod(putMessage);
			log.info("response from social networking service is {}",returnedStatus);
			return returnedStatus>=200 && returnedStatus<=205;
		} catch (Exception e) {
			log.error("error trying to send invitation",e);
			throw new Exception("error trying to send invitation",e);
		} 


	}

	private String retrieveSocialServiceEnpoint() throws Exception {
		XQuery query = queryFor(GCoreEndpoint.class);
		query.addCondition("$resource/Profile/ServiceClass/text() eq 'Portal'");
		query.addCondition("$resource/Profile/ServiceName/text() eq 'SocialNetworking'");
		query.addVariable("$entry","$resource/Profile/AccessPoint/RunningInstanceInterfaces/Endpoint");
		query.addCondition("$entry/@EntryName/string() eq 'jersey-servlet'");
		query.setResult("$entry/text()");
		DiscoveryClient<String> client = client();

		List<String> socialServiceEnpoints = client.submit(query);

		if (socialServiceEnpoints.size()==0) throw new Exception("Social servioce enpooint not found in the current scope "+ScopeProvider.instance.get()); 

		String socialServiceEnpoint = socialServiceEnpoints.get(0);

		return socialServiceEnpoint+"/2";
	}

	private String getGeneratadePassword(){
		return "gcube_SL_"+Math.abs(AuthorizationProvider.instance.get().getClient().getId().hashCode());
	}

	private MongoClient connectToMongo(){
		String mongoHost = context.getInitParameter("mongoHost");
		int mongoPort = Integer.parseInt(context.getInitParameter("mongoPort"));

		/*String mongoLogin = context.getInitParameter("mongoLogin");
		String mongoPassword = context.getInitParameter("mongoPwd");
		String mongoDB = context.getInitParameter("mongoDb");*/

		//MongoClientOptions options = MongoClientOptions.builder().sslEnabled(true).build();
		MongoClient mongoClient = new MongoClient(new ServerAddress(mongoHost, mongoPort));

		return mongoClient;
	}
}
