package org.gcube.data.access.httpproxy.access;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.gcube.data.access.httpproxy.utils.Properties;
import org.gcube.data.access.httpproxy.utils.Properties.LongPropertyType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public  class URLCache {

	private Map<String, List<String>> domainMap;
	private static URLCache instance;
	private final Lock readLock;
	private final Lock writeLock;
	private Logger logger;
	private Map<String, Long> expiringTimes;
	private ISManager isManager;
	private long timerPeriod;
	
	private URLCache ()
	{
		this.logger = LoggerFactory.getLogger(this.getClass());
		this.logger.debug("Generating new cache instance");
		this.domainMap = new HashMap<>();
		this.expiringTimes = new HashMap<>();
		this.isManager = new ISManager();
		this.timerPeriod = Properties.getInstance().getProperty(LongPropertyType.DOMAIN_FILTER_DEFAULT_EXPIRING_TIME)*1000;
		this.logger.debug("Expiring period "+this.timerPeriod);
		ReadWriteLock readWriteLock = new ReentrantReadWriteLock();
		this.readLock = readWriteLock.readLock();
		this.writeLock = readWriteLock.writeLock();
		this.logger.debug("New cache instance generated");
	}

	
	public static URLCache getInstance ()
	{
		if (instance == null) instance = new URLCache();
		
		return instance;
	}
	
	
	public  void  setDomainList (String scope, List<String> domains)
	{
		this.logger.debug("Locking domain list for writing operations");
		this.writeLock.lock();
		
		try
		{
			this.logger.debug("Writing new domain list for thread "+Thread.currentThread().getName());
			
			if (domains == null) domains = new ArrayList<>();
			
			this.domainMap.put(scope, domains);
			this.expiringTimes.put(scope, new Date().getTime()+this.timerPeriod);
			this.logger.debug("Writing operation completed");
		}
		finally
		{
			
			this.writeLock.unlock();
			this.logger.debug("Object unlocked");
		}

	}
	
	public List<String> getDomainList (String scope)
	{
		refreshDomains(scope);
		this.readLock.lock();
		
		try
		{
			this.logger.debug("Reading new domain list for thread "+Thread.currentThread().getName());
			return this.domainMap.get(scope);
		}
		finally
		{
			this.readLock.unlock();
			this.logger.debug("Object unlocked");
		}

	}
	
	private void refreshDomains (String scope)
	{
		this.logger.debug("Evaluating if time is to be refreshed");
		long currentTime = new Date().getTime();
		this.logger.debug("Current time "+currentTime);
		Long expiringTime = this.expiringTimes.get(scope);
		
		if (expiringTime == null || currentTime > expiringTime)
		{
			this.logger.debug("Expiring time "+expiringTime);
			this.logger.debug("Domain list expired");
			loadDomains(scope);
			
		}
		else this.logger.debug("Domain list valid");
	}

	private void loadDomains (String scope)
	{
		logger.debug("Asking IS for a new List...");
		
		try
		{
			List<String> domains = isManager.getDomains();
			logger.debug("List obtained");
			this.logger.debug("List "+domains);
			this.logger.debug("Adding new list to the cache for scope "+scope);
			setDomainList(scope, domains);
			this.logger.debug("Operation completed");
		}
		catch (RuntimeException e)
		{
			this.logger.warn("Unable to refresh the list: using the current one",e);
			this.readLock.lock();
			List<String> domains = null;
			
			try
			{
				
				domains = this.domainMap.get(scope);
				
			} finally
			{
				this.readLock.unlock();
						
			}
			
			
			if (domains == null)
			{
				setDomainList(scope, null);
			}
			
		}
		

	}


}
