package org.gcube.data.access.httpproxy;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.IOUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.gcube.data.access.httpproxy.base.URIModifier;
import org.gcube.data.access.httpproxy.base.URIModifierFactory;
import org.gcube.data.access.httpproxy.utils.LogUtils;
import org.gcube.data.access.httpproxy.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class ProxyServlet extends HttpServlet {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	private Logger logger;

	
	public ProxyServlet() {
	 super();
	 this.logger = LoggerFactory.getLogger(this.getClass());

	}
	

	private HttpPost preparePostRequest (URI finalAddress,HttpServletRequest originalpostRequest) throws IOException
	{
		HttpPost httpPost = new HttpPost(finalAddress);
		String contentType = originalpostRequest.getContentType();
		logger.debug("Content type = "+contentType);
		httpPost.setHeader("Content-Type", contentType);
		String accept = originalpostRequest.getHeader("Accept");
		logger.debug("Accept = "+accept);
		httpPost.setHeader("Accept", accept);
		String acceptEncoding = originalpostRequest.getHeader("Accept-Encoding");
		logger.debug("Accept Encoding = "+acceptEncoding);
		httpPost.setHeader("Accept-Encoding", acceptEncoding);
		String authorization = originalpostRequest.getHeader("Authorization");
		logger.debug("Authorization = "+authorization);
		httpPost.setHeader("Authorization", authorization);	
		return httpPost;

	}
	
	private String parseResponseEntity (HttpEntity responseEntity,String responseContentType,HttpServletRequest request) throws UnsupportedOperationException, IOException
	{
		logger.debug("Entity found");
		InputStream in = responseEntity.getContent();
		StringWriter writer = new StringWriter();
		IOUtils.copy(in, writer,"UTF-8");
		String html = writer.toString();
		writer.close();
		URIModifier uriModifier = URIModifierFactory.getInstance().getUriModifier(responseContentType);
		uriModifier.init(request);
		return uriModifier.modify(html);

	}
	
	private void printData (HttpServletResponse servletResponse,String responseContentType, String data) throws IOException
	{
		logger.debug("Printing data");
		servletResponse.setContentType(responseContentType);
		PrintWriter out = servletResponse.getWriter();
		out.println(data);
		 
		logger.debug("Data printed");
		out.close();
	}

	private void forwardMessage (HttpUriRequest request,HttpServletRequest servletRequest,HttpServletResponse servletResponse) throws ClientProtocolException, IOException
	{
		CloseableHttpClient httpClient = HttpClients.createDefault();
		CloseableHttpResponse response = httpClient.execute(request);
		servletResponse.setStatus(response.getStatusLine().getStatusCode());
		String responseContentType = response.getFirstHeader("Content-Type").getValue();
		logger.debug("Response Content type = "+responseContentType);
		LogUtils.logHeaders("Server Response", this.logger, response);
		HttpEntity responseEntity = response.getEntity();
		servletResponse.setContentType(responseContentType);
		
		if (responseEntity != null)
		{
		
			String html = parseResponseEntity(responseEntity, responseContentType, servletRequest);
			printData(servletResponse, responseContentType, html);
			
		}

	}

	
	private void sendErrorMessage (HttpServletResponse resp) throws IOException
	{
		// Set response content type
		resp.setContentType("text/html");
		// Actual logic goes here.
		PrintWriter out = resp.getWriter();
		out.println("<h1>Please set a correct address</h1>");
	}
	
	private void forwardPostRequest (URI finalAddress, HttpServletRequest request,HttpServletResponse servletResponse) throws Exception
	{
		HttpPost httpPost = preparePostRequest(finalAddress, request);

		InputStream requestStream = request.getInputStream();
		
		if (requestStream != null)
		{
			logger.debug("Adding input stream");
			InputStreamEntity entity = new InputStreamEntity(requestStream);
			httpPost.setEntity(entity);
			logger.debug("Input stream added");
			
		}

		forwardMessage(httpPost, request, servletResponse);


		if (requestStream != null) requestStream.close(); 
	}
	



	@Override
	public void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException 
	{

		logger.debug("Forwarding request to the final address");
		String path = req.getPathInfo();
		logger.debug("Path = "+path);
		
		try
		{
			
			URI finalAddress = Utils.getAddress(path, req, true);
			logger.debug("Final address "+finalAddress);
			forwardPostRequest(finalAddress, req,resp);
			
			
		} catch (URISyntaxException use)
		{
			sendErrorMessage (resp);
		}
		catch (Exception e)
		{
			logger.error("Unable to serve the request",e);
			throw new ServletException(e);
		}
		

	}
	
	
	
	@Override
	public void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException 
	{

		logger.debug("Forwarding request to the final address");
		String path = req.getPathInfo();
		logger.debug("Path = "+path);
		
		try
		{
			URI finalAddress = Utils.getAddress(path, req, true);
			logger.debug("Final address "+finalAddress);
			HttpGet httpGet = new HttpGet(finalAddress);
			forwardMessage(httpGet, req, resp);
			
		} catch (URISyntaxException use)
		{
			sendErrorMessage (resp);
		}
		catch (Exception e)
		{
			logger.error("Unable to serve the request",e);
			throw new ServletException(e);
		}
		
	
		


	}
	
	

	public static void main(String[] args) throws Exception{
		Logger logger = LoggerFactory.getLogger(ProxyServlet.class);
		
	//	ProxyAuthenticator authenticator = new ProxyAuthenticator();
		
		
//		if (authenticator.isActive())
//			{
//			logger.debug("Setting proxy");
//			Authenticator.setDefault(authenticator);
//			}
//		
		

		CloseableHttpClient httpClient = HttpClients.createDefault();
		HttpGet httpGet = new HttpGet("http://www.eng.it");
		CloseableHttpResponse response = httpClient.execute(httpGet);
		System.out.println("Status = "+response.getStatusLine());

		Header [] headers = response.getAllHeaders();
		
		for (Header header : headers)
		{
			String headerName = header.getName();
			String headerValue = header.getValue();
			logger.debug("Header name "+headerName);
			logger.debug("Header value "+headerValue);
	
		}
		
		
		HttpEntity responseEntity = response.getEntity();
		logger.debug("Entity found");
		InputStream in = responseEntity.getContent();
		
		List<String> lines = IOUtils.readLines(in);
		
		for (String line : lines)
		{
			System.out.println(line);
		}
		
		
	}

}
