/**
 * 
 */
package org.gcube.data.gml.elements;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Calendar;
import java.util.Map;

/**
 * The read-only interface of all elements of the gCube Document Model, 
 * including {@link GCubeDocument}s and their inner elements.
 * 
 * @author Federico De Faveri defaveri@isti.cnr.it
 * @author Fabio Simeoni (University of Strathclyde)
 */
public interface GCubeElement {
	
	/**
	 * Returns the identifier of the element.
	 * @return the identifier.
	 */
	String id();
	
	/**
	 * Returns the descriptive name of the element.
	 * @return the name.
	 */
	String name();
	
	/**
	 * Returns the type of the element.
	 * @return the name.
	 */
	String type();
	
	/**
	 * Returns the creation time of the element.
	 * @return the creation time.
	 */
	Calendar creationTime();
	
	/**
	 * Returns the time of last update of the element.
	 * @return the last update time.
	 */
	Calendar lastUpdate();
	
	/**
	 * Returns the Mime type of the element.
	 * @return the Mime type.
	 */
	String mimeType();
	
	/**
	 * Returns the length of the element's bytestream.
	 * @return the length.
	 */
	Long length();
	
	/**
	 * Returns the element's bytestream.
	 * @return the bytestream, or <code>null</code> if the element has no content or does not include it.
	 */
	byte[] bytestream();
	
	/**
	 * Returns a {@link URI} to the element's bytestream.
	 * @return the URI, or <code>null</code> if a URI for the element's bytestream does
	 * not exist or is not included in the element.
	 */
	URI bytestreamURI();
	
	/**
	 * Returns an {@link InputStream} over the element's bytestream, resolving it if it is referenced by a {@link URI}.
	 * @return the stream, or <code>null</code> if the element has no bytestream.
	 */
	public InputStream resolveBytestream() throws IOException;
	
	/**
	 * Returns the storage URI of the element.
	 * @return the URI.
	 * @throws IllegalStateException if the element has no identifier or is not bound to a document.
	 * @throws URISyntaxException if the identifier of the element or its document cannot be used in a URI.
	 */
	URI uri() throws IllegalStateException, URISyntaxException;

	
	/**
	 * Returns the language of the element.
	 * @return the language.
	 */
	String language();
	
	
	/**
	 * Return the URI of the element's schema.
	 * @return the URI.
	 */
	URI schemaURI();

	/**
	 * Returns the descriptive name of the element's schema.
	 * @return the name.
	 */
	String schemaName();

	/**
	 * Returns a generic property of the element.
	 * @param key the property key.
	 * @return the property value, or <code>null</code> if the element has no property with the given key.
	 */
	GCubeElementProperty property(String key);
	
	/**
	 * Returns the generic properties of the element.
	 * @return the properties.
	 */
	Map<String, GCubeElementProperty> properties();
	
}
